(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    452740,       8838]
NotebookOptionsPosition[    445379,       8607]
NotebookOutlinePosition[    445866,       8624]
CellTagsIndexPosition[    445823,       8621]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["Chapter 5", "Title",
 Editable->False,
 CellChangeTimes->{{3.393757273997875*^9, 3.393757275404125*^9}, 
   3.3998859388906655`*^9, 3.4119881486574545`*^9}],

Cell["Poisson Fibre Processes II: Void Phase", "Subtitle",
 Editable->False,
 CellChangeTimes->{{3.3979872211263776`*^9, 3.3979872252523108`*^9}, 
   3.4119880476021442`*^9, {3.4124102745954733`*^9, 3.4124103183483872`*^9}}],

Cell["\[Copyright]W.W.Sampson, Manchester, 2008", "Subsubtitle",
 Editable->False,
 CellChangeTimes->{{3.412935407213834*^9, 3.412935418490048*^9}}],

Cell[CellGroupData[{

Cell["5.2 In-Plane Pore Dimensions", "Section",
 Editable->False,
 CellChangeTimes->{{3.4124103371153727`*^9, 3.4124103438750925`*^9}}],

Cell["\<\
The expected in-plane pore diameter is very sensitive to porosity, \
\[Epsilon], at high and low porosities and approximately log-linear for \
porosities between about 0.2 and 0.7:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.4124103710341454`*^9, 3.4124104348258734`*^9}, 
   3.4124158040063763`*^9}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"dbar", " ", "=", " ", 
   RowBox[{"2", " ", 
    RowBox[{"\[Omega]", "/", 
     RowBox[{"Log", "[", 
      RowBox[{"1", "/", "\[Epsilon]"}], "]"}]}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{"LogPlot", "[", 
  RowBox[{
   RowBox[{"dbar", "/", "\[Omega]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"\[Epsilon]", ",", " ", "0", ",", " ", "1"}], "}"}], ",", " ", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
     "\"\<\[Epsilon]\>\"", ",", " ", 
      "\"\<\!\(\*OverscriptBox[\(d\), \(_\)]\)/\[Omega]\>\""}], "}"}]}]}], 
  "]"}]}], "Input",
 CellChangeTimes->{{3.40144194646875*^9, 3.40144204321875*^9}, {
   3.401442078640625*^9, 3.401442096140625*^9}, {3.401449940609375*^9, 
   3.40144994915625*^9}, {3.401528733140506*^9, 3.4015287372464094`*^9}, 
   3.40152898291967*^9, {3.4304762351838384`*^9, 3.4304762358447886`*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwVj3k81Pn/wFEY5xhT27qSQq4kWaXD6+2rcuYuKWEiotZRWlLOReVoU1YW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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox["\"\[Epsilon]\"", TraditionalForm], 
    FormBox[
    "\"\\!\\(\\*OverscriptBox[\\(d\\), \\(_\\)]\\)/\[Omega]\"", 
     TraditionalForm]},
  AxesOrigin->{0, -2.180831545933616},
  FrameTicks->{{{{0, 
       FormBox[
        TagBox[
         InterpretationBox["\"1.0\"", 1., AutoDelete -> True], 
         NumberForm[#, {
           DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
        TraditionalForm]}, {
       NCache[-Log[2], -0.6931471805599453], 
       FormBox[
        TagBox[
         InterpretationBox["\"0.5\"", 0.5, AutoDelete -> True], 
         NumberForm[#, {
           DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
        TraditionalForm]}, {
       NCache[
        Log[5], 1.6094379124341003`], 
       FormBox[
        TagBox[
         InterpretationBox["\"5.0\"", 5., AutoDelete -> True], 
         NumberForm[#, {
           DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
        TraditionalForm]}, {
       NCache[-Log[10], -2.302585092994046], 
       FormBox[
        TagBox[
         InterpretationBox["\"0.1\"", 0.1, AutoDelete -> True], 
         NumberForm[#, {
           DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
        TraditionalForm]}, {
       NCache[
        Log[10], 2.302585092994046], 
       FormBox[
        TagBox[
         InterpretationBox["\"10.0\"", 10., AutoDelete -> True], 
         NumberForm[#, {
           DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
        TraditionalForm]}, {
       NCache[
        Log[50], 3.912023005428146], 
       FormBox[
        TagBox[
         InterpretationBox["\"50.0\"", 50., AutoDelete -> True], 
         NumberForm[#, {
           DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
        TraditionalForm]}, {
       NCache[
        Log[100], 4.605170185988092], 
       FormBox[
        TagBox[
         InterpretationBox["\"100.0\"", 100., AutoDelete -> True], 
         NumberForm[#, {
           DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
        TraditionalForm]}, {
       NCache[
        Log[500], 6.214608098422191], 
       FormBox[
        TagBox[
         InterpretationBox["\"500.0\"", 500., AutoDelete -> True], 
         NumberForm[#, {
           DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
        TraditionalForm]}, {-1.6094379124341003`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-1.203972804325936, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.916290731874155, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.5108256237659907, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.35667494393873245`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.2231435513142097, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.10536051565782628`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {0.6931471805599453, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.0986122886681098`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.3862943611198906`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.791759469228055, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.9459101490553132`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {2.0794415416798357`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {2.1972245773362196`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {2.995732273553991, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {3.4011973816621555`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {3.6888794541139363`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.0943445622221, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.248495242049359, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.382026634673881, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.499809670330265, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {5.298317366548036, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {5.703782474656201, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {5.991464547107982, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}}, {{0, 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[-Log[2], -0.6931471805599453], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[
        Log[5], 1.6094379124341003`], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[-Log[10], -2.302585092994046], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[
        Log[10], 2.302585092994046], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[
        Log[50], 3.912023005428146], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[
        Log[100], 4.605170185988092], 
       FormBox["\" \"", TraditionalForm]}, {
       NCache[
        Log[500], 6.214608098422191], 
       FormBox["\" \"", TraditionalForm]}, {-1.6094379124341003`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-1.203972804325936, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.916290731874155, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.5108256237659907, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.35667494393873245`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.2231435513142097, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.10536051565782628`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {0.6931471805599453, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.0986122886681098`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.3862943611198906`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.791759469228055, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.9459101490553132`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {2.0794415416798357`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {2.1972245773362196`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {2.995732273553991, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {3.4011973816621555`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {3.6888794541139363`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.0943445622221, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.248495242049359, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.382026634673881, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.499809670330265, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {5.298317366548036, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {5.703782474656201, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {5.991464547107982, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}}}, {Automatic, Automatic}},
  GridLines->{None, None},
  Method->{},
  PlotRange->{{0, 1}, {-2.180831545933616, 5.184098830998777}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]},
  Ticks->{Automatic, {{0, 
      FormBox[
       TagBox[
        InterpretationBox["\"1.0\"", 1., AutoDelete -> True], NumberForm[#, {
          DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
       TraditionalForm]}, {
      NCache[-Log[2], -0.6931471805599453], 
      FormBox[
       TagBox[
        InterpretationBox["\"0.5\"", 0.5, AutoDelete -> True], 
        NumberForm[#, {
          DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
       TraditionalForm]}, {
      NCache[
       Log[5], 1.6094379124341003`], 
      FormBox[
       TagBox[
        InterpretationBox["\"5.0\"", 5., AutoDelete -> True], NumberForm[#, {
          DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
       TraditionalForm]}, {
      NCache[-Log[10], -2.302585092994046], 
      FormBox[
       TagBox[
        InterpretationBox["\"0.1\"", 0.1, AutoDelete -> True], 
        NumberForm[#, {
          DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
       TraditionalForm]}, {
      NCache[
       Log[10], 2.302585092994046], 
      FormBox[
       TagBox[
        InterpretationBox["\"10.0\"", 10., AutoDelete -> True], 
        NumberForm[#, {
          DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
       TraditionalForm]}, {
      NCache[
       Log[50], 3.912023005428146], 
      FormBox[
       TagBox[
        InterpretationBox["\"50.0\"", 50., AutoDelete -> True], 
        NumberForm[#, {
          DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
       TraditionalForm]}, {
      NCache[
       Log[100], 4.605170185988092], 
      FormBox[
       TagBox[
        InterpretationBox["\"100.0\"", 100., AutoDelete -> True], 
        NumberForm[#, {
          DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
       TraditionalForm]}, {
      NCache[
       Log[500], 6.214608098422191], 
      FormBox[
       TagBox[
        InterpretationBox["\"500.0\"", 500., AutoDelete -> True], 
        NumberForm[#, {
          DirectedInfinity[1], 1}, NumberPadding -> {"", "0"}]& ], 
       TraditionalForm]}, {-1.6094379124341003`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-1.203972804325936, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-0.916290731874155, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-0.5108256237659907, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-0.35667494393873245`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-0.2231435513142097, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-0.10536051565782628`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {0.6931471805599453, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {1.0986122886681098`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {1.3862943611198906`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {1.791759469228055, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {1.9459101490553132`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {2.0794415416798357`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {2.1972245773362196`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {2.995732273553991, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {3.4011973816621555`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {3.6888794541139363`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {4.0943445622221, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {4.248495242049359, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {4.382026634673881, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {4.499809670330265, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {5.298317366548036, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {5.703782474656201, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {5.991464547107982, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}}}]], "Output",
 CellChangeTimes->{
  3.401442018296875*^9, 3.4014420966875*^9, 3.40144995384375*^9, 
   3.40145299328125*^9, 3.401454512078125*^9, 3.40145462528125*^9, 
   3.4015266747006145`*^9, {3.401528739790067*^9, 3.4015287561636114`*^9}, 
   3.4015289971901903`*^9, 3.401613541228547*^9, 3.4016139073049393`*^9, 
   3.401615848315976*^9, 3.4016160981151695`*^9, 3.4017099593409843`*^9, 
   3.411302215552725*^9, 3.4304762691026115`*^9, {3.430480879902616*^9, 
   3.4304808974979167`*^9}},
 ImageCache->GraphicsData["CompressedBitmap", "\<\
eJztm3uMHVUdx6d77+1uLViEUkprYUtbKEUetl1FCxbQtlugUpVXJARLQ9ia
VBqoRnykJSYlISJRYrBJY01AUSHZGkwJEaxKQF6hCdpAhIj11UatRAnRqvG6
n99vzpwzc89vZnZ7l+UPb/bOvTPn/L6/9+887tk16zYP3bBx3eYN69f1r7p5
3aahDetv6V95080jjxqTkmTSUJIkB/sTvrdHvqaX9shfewMXnsS/f4yPHro3
5EEjUVJB0LulhS6trIt78xowgXj31OwivOR5j9XlGGX5Uflo5tpHXq2IFkvi
XSdHOCyOd23GHxvARu+lEX6NeNdppTqOVvDm6GWdIM16I0DvLhMl7+il3dCx
lUXpwASo5bmXKdOo6jCOPhqrNQ1xiwnLe7yjzxe9yXln91jeWpJv9QoVUGLP
0oIVVt1ZkpaIJE8vCZ4mjXbIWK6Dmc8nVfRcFZG9pw5NM0CfzocYbHXbsEcn
Zv+BA3+Xu6MjfI6KPIv1q/vsSPDeSrK8aboZwVPqtHdOkBITxffNcMY7bBec
F+l+JM+WjY69dLj0/+1da+/Jyvuat5BU2r5w//7Xytov2Pv8H46kffDpp35b
1v6Jh3e/NHqr3XX3154sQx3+/OceKWvfu/76B8va/7Higzv6Dh/+j9W+56ZP
PaSax9s3PfD9F1RG2yuvXnXFdwytkwYCnvPyr/7ihbw4BGm6XF77+M9eRdnO
yUMrTHcPF0BNFX7CTiVLHAUvaWh1Qp2775cHf75haFcO6m0qVRL046s9X5ic
n3y027dv/8bTn/32fXvlZvUYUXuLqPgQYb3ug2ODVhM0jnn99cPgadbQsKoU
z895W975WYfU7AQDAaWpyrOV1aAFIZsR+KaHz9tgRdH/fiYbw0n5zTx06A1w
NGN59qGOuM5odGHaxP47b9/2E97YThovKob7FAUA+MVrr/melgyeXWgzUJom
Qj386c27yQEUlcblNtlUJSNtYEXUZXKdb5PpAN5CHYKU1N2685vPQaqSLivq
pOuLFj3oCTOKgi8q77O56TKhhW4UkQMfuexePn18vNemPdbzhR+SYnzv/IGi
pDOUAutTJngP/WDXPq/bYpvbCUoL+j13fvlxJIWbL43n2LQnaqBgEXiTAUiL
fTNPnlkUdo4S0QHfQYDAqJoRLQrCu3wDKZaUJysHVMLocEA0TIJDpNNpyuGo
CK6x3Izl1SnKCWM54zldqC/Saf7YOEm5mpfXhBB0Mc8zoZjbFU0WyrUXqXEf
WrhQuPqxR1/xoTSnK+zOUHa4g8i5/7atj752yeqdFCfYM0Ipu1ldYXe2sgMW
28EGdrCFPWIouxlB4BVW/rFI07zqxTpYCWu5GICPL4PH2Un0HsXAn6hOKWwv
P387EZs3xTQb4/2KQRyG+lFSiUVfOqbaGFo7+9CF2kq8oUcRRzr32jAXlsM4
ldqa2ibMijyM04pZHRZyMP+1EXQG0kdxwg8OwdmWko69/mUjfFgR6EVvqKAG
BTTkInLesBF0160PSZEYAxAf5DCxQuyRcX+zEa5UBHrRGyqoQQnNechG+Lhc
pyBpaMnQDthHhrWDNsx1eWPgBJyBb/Ex4U/5/r2NsF4RCCKCieSjNBdV+Y2N
cGO5KqFTkpcDGBl3NnoFQl+44ZXRAdFetNlvUgTUxBdUfBfamMNF1C9shM2q
AFGNzYtRhSEzBZ63YW71YYUli2HlVHnGRviiCuLCylJFguIJG+a2uD4EhtOH
wBB9fmrDbMvrQ2TEXPOYjXBHuWtwOPo8YiN8Ja8KseRc05EmP7Rh7o6nSehd
WOyyEe7pdC5quERzCA8W43tHPr1COszh/PBdm/NOj4DJnRsIqxDhPhvhXl+t
yhC+ZSPcbyOE2u+wER6Qq8xkQju4qi1j8nabfFjJXcl2w7GLIhlKv26TP6Tk
LoTCoQ99ZAHwVZt8t6rvhj2MRhY4A/KMCLyz6PwfeaVhAzvCzk0iROo7bLY/
VnI3f4jO/rbZ5Jra5mxOuH9JyWVBLPM482fj2ETrCe+VooDOJskW5XB0FJdr
raniU3ldXPjwyb2Y4gtd4fSsckJ2dEAXOLmlioTKrWPjJJtLOoBki5NwoUWc
fEYp/FS3VV/0FxS6uC6h4gF9SwAd3eWoxWSfMgHQrb9DJjoa68rAy29G0Evl
aBuLGfVKvL9b237SzoZfe0q3wMWnLnputCn3y7WFx9xSgk8pWTfYZL9Thm4Z
TqmCKcwR9Xqb8o9KSS+3KeJEBe26olH+1CkgoSV7adfabP6sZGhPZGNHhKS0
JNfYZH9VMtR34wCukNnE1UXRdA4tOy50cnMHEe1Km4furLec7ZxoYvLLbTKd
8zfpRnfIINf5vi4a/b5F+SmZVqbCPxUUkREdFQBdq6C98cyJ74zUTrLDqj8R
4Hhmm1xrxpPxv7UrjN0mDYwv7SpP2URJObm9OGfWi8dTu7bXDrNSd6nxg+PO
M+1PZWbIYggmW1cGgenDscYOh1zTos3oS3kgGbFmyeaxXNMdZNLcbQsSVxdV
kKXWoasb68mzkp1kuU7x2eO2kxFyeQVZOki57WTZGv9ABY3uRjWpZ3gXdrA6
r4Ls7dqAMtkm+bIKmmmeFWzcLnzJ1rNc9bfh/H78uRU0uu/ccPvxVNGkZJta
rtO1gb4YXGZHAxU0x3uLE54yz1laQaP7cg2cSzzI7vSSChrd2W4408loU7IR
LteZ2kDsEeVUpPRwV/EXzdnak4oiluLu7Ar02d5aRLbcnVVBkx6gIHlwvtyd
WUEzx3s/o3lXBc1J2uB/9my7XVqb5mRtcD/8yd2iCpp+bcC64hHuTq9Hg1fw
pHi/imau9z4RI1G2sIJGt/MbFE8iUwbB0ypo5kVsfWo0YhZoT7fxJ3cLKtDn
e83JY9GiimZBRIv5FTSnRjw5rx4N+FhYZlpVNGrNBh5kJJG7UypoFkYiZm49
GmoZQ4/YIKTxXlnk8yvLyf4K9NO95qDLhLSKZlEkvzR/dNz1a+Ia08USBjmh
UgZ9BbBuTTbOiOh0UvdYNotOyunWRUYx3aRe9lCqU0Z+nlQKNUoOhRidE/Ay
zuZ3QzkdQBqs3ViQyZ0ONMFBAr92qTFFTBH90RKPWEg4HSQblLPs5NLsiuQ5
K5KkNWkwLyvHWjSpbNkKkLtZUS30V/F82Z9VD50hiY2eWjQpH39aqe1+dq+k
yVnrxKgWOrWRoSI7T1WFntIQstnJtJn1aHJDX0qTBmnwKqyZ666e5aHO7xqM
E4wXISMjR4rp1K0US1X2x8QmTJLFEYedMO6StIqpknP++PMvsQQlj9I38ZLk
Ei+VJI119yrkQN16rIujhj8FOXI3oyJLUxrqt546qE/D8C+LpVHQ5NxQkyZn
sJo06I8datGk/3+SG52Or0eTGzVq0uQm3DVpcmNOTZrchLsmTa50VdEM+DjI
GqZX0HQeqrXby44SR6nke8X/S9b5nu4iJB18LtP2mVa7nh7RU/otXzfyNaKd
/rOOfFsdUHTWEuO4U6xCSeNgiBYZOeMnhsvBji0TLU9W/Je+zC7pVqSciAp+
CKwELQKkhx6L5vQypw6QyDUnEVbIpGa7YM+ePVu3bNnCO2node3w8HA0FEL6
ZNL/AA7rNsc=\
\>"]]
}, Open  ]],

Cell["We assume a gamma probability density of pore diameters:", "Text",
 Editable->False,
 CellChangeTimes->{{3.4124104489662066`*^9, 3.4124104789593344`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"pdfd", " ", "=", " ", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"GammaDistribution", "[", 
     RowBox[{"\[Alpha]", ",", 
      RowBox[{"\[Beta]", "/", "\[Alpha]"}]}], "]"}], ",", "d"}], 
   "]"}]}], "\[IndentingNewLine]", 
 RowBox[{"cdfd", " ", "=", " ", 
  RowBox[{"FunctionExpand", "[", 
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"GammaDistribution", "[", 
      RowBox[{"\[Alpha]", ",", 
       RowBox[{"\[Beta]", "/", "\[Alpha]"}]}], "]"}], ",", "d"}], "]"}], 
   "]"}]}]}], "Input",
 CellChangeTimes->{{3.401451952640625*^9, 3.4014519615625*^9}, {
  3.401452049265625*^9, 3.4014520911875*^9}, {3.401452160453125*^9, 
  3.401452176734375*^9}, {3.401454493796875*^9, 3.40145449928125*^9}, {
  3.401454612015625*^9, 3.401454613484375*^9}}],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["d", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Alpha]"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox[
      RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}]], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox["\[Beta]", "\[Alpha]"], ")"}], 
    RowBox[{"-", "\[Alpha]"}]]}], 
  RowBox[{"Gamma", "[", "\[Alpha]", "]"}]]], "Output",
 CellChangeTimes->{
  3.401452993421875*^9, 3.401454512234375*^9, 3.4014546254375*^9, 
   3.4015266750511184`*^9, 3.401528756504101*^9, 3.4015289975306797`*^9, 
   3.4016135732345695`*^9, 3.4016139076254*^9, 3.401615848536293*^9, 
   3.4016160983655295`*^9, 3.401709960021963*^9, 3.411302215923258*^9, 
   3.430477012912157*^9, {3.4304808802030478`*^9, 3.430480897698205*^9}}],

Cell[BoxData[
 RowBox[{"1", "-", 
  FractionBox[
   RowBox[{"Gamma", "[", 
    RowBox[{"\[Alpha]", ",", 
     FractionBox[
      RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}], "]"}], 
   RowBox[{"Gamma", "[", "\[Alpha]", "]"}]]}]], "Output",
 CellChangeTimes->{
  3.401452993421875*^9, 3.401454512234375*^9, 3.4014546254375*^9, 
   3.4015266750511184`*^9, 3.401528756504101*^9, 3.4015289975306797`*^9, 
   3.4016135732345695`*^9, 3.4016139076254*^9, 3.401615848536293*^9, 
   3.4016160983655295`*^9, 3.401709960021963*^9, 3.411302215923258*^9, 
   3.430477012912157*^9, {3.4304808802030478`*^9, 3.430480897858435*^9}}]
}, Open  ]],

Cell["\<\
The probability density of the smallest of two such pores is:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.412410494331438*^9, 3.412410515131347*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"pdf2layers", " ", "=", " ", 
  RowBox[{"2", 
   RowBox[{"(", 
    RowBox[{"1", "-", "cdfd"}], ")"}], "pdfd"}]}]], "Input",
 CellChangeTimes->{{3.401452281109375*^9, 3.40145233828125*^9}, {
  3.40145358940625*^9, 3.40145361434375*^9}, {3.40145476728125*^9, 
  3.40145477146875*^9}}],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", " ", 
   SuperscriptBox["d", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Alpha]"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox[
      RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}]], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox["\[Beta]", "\[Alpha]"], ")"}], 
    RowBox[{"-", "\[Alpha]"}]], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"\[Alpha]", ",", 
     FractionBox[
      RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}], "]"}]}], 
  SuperscriptBox[
   RowBox[{"Gamma", "[", "\[Alpha]", "]"}], "2"]]], "Output",
 CellChangeTimes->{
  3.40145477878125*^9, 3.4015266767335377`*^9, 3.4015287572351522`*^9, 
   3.401528998552149*^9, 3.4016135741458797`*^9, 3.4016139293666625`*^9, 
   3.401615849117128*^9, 3.40161610102936*^9, 3.401709961263749*^9, 
   3.411302237774678*^9, 3.430477015545944*^9, {3.430480883768174*^9, 
   3.430480897898493*^9}}]
}, Open  ]],

Cell["So the cumulative distribution function is:", "Text",
 Editable->False,
 CellChangeTimes->{{3.4124105211199584`*^9, 3.41241053528032*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"cdf2layers", " ", "=", " ", 
  RowBox[{"Simplify", "[", 
   RowBox[{"PowerExpand", "[", 
    RowBox[{"Integrate", "[", 
     RowBox[{"pdf2layers", ",", " ", 
      RowBox[{"{", 
       RowBox[{"d", ",", " ", "0", ",", " ", "d"}], "}"}], ",", " ", 
      RowBox[{"Assumptions", "\[Rule]", 
       RowBox[{
        RowBox[{"Re", "[", "\[Alpha]", "]"}], ">", "0"}]}]}], "]"}], "]"}], 
   "]"}]}]], "Input",
 CellChangeTimes->{{3.401452401421875*^9, 3.40145241325*^9}, {
  3.401452521859375*^9, 3.40145252946875*^9}, {3.4014526429375*^9, 
  3.40145265121875*^9}}],

Cell[BoxData[
 RowBox[{"1", "-", 
  FractionBox[
   SuperscriptBox[
    RowBox[{"Gamma", "[", 
     RowBox[{"\[Alpha]", ",", 
      FractionBox[
       RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}], "]"}], "2"], 
   SuperscriptBox[
    RowBox[{"Gamma", "[", "\[Alpha]", "]"}], "2"]]}]], "Output",
 CellChangeTimes->{3.40145505959375*^9, 3.401526732694005*^9, 
  3.401528819464634*^9, 3.4015290572565613`*^9, 3.4016135950559473`*^9, 
  3.4016139845860643`*^9, 3.401615900951662*^9, 3.4016161537752047`*^9, 
  3.401710014269968*^9, 3.4113022946764994`*^9, 3.430477070054323*^9, 
  3.430480885230277*^9, 3.4304809478603344`*^9}]
}, Open  ]],

Cell["\<\
Similarly, the smallest of three such pores has probability density\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.4124105422002707`*^9, 3.4124105632605534`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"pdf3layers", " ", "=", " ", 
  RowBox[{"3", 
   RowBox[{"(", 
    RowBox[{"1", "-", "cdf2layers"}], ")"}], "pdfd"}]}]], "Input",
 CellChangeTimes->{{3.401526837024024*^9, 3.4015268526164446`*^9}}],

Cell[BoxData[
 FractionBox[
  RowBox[{"3", " ", 
   SuperscriptBox["d", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Alpha]"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox[
      RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}]], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox["\[Beta]", "\[Alpha]"], ")"}], 
    RowBox[{"-", "\[Alpha]"}]], " ", 
   SuperscriptBox[
    RowBox[{"Gamma", "[", 
     RowBox[{"\[Alpha]", ",", 
      FractionBox[
       RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}], "]"}], "2"]}], 
  SuperscriptBox[
   RowBox[{"Gamma", "[", "\[Alpha]", "]"}], "3"]]], "Output",
 CellChangeTimes->{3.401526873396325*^9, 3.4015271601486545`*^9, 
  3.40152881952472*^9, 3.401529057416792*^9, 3.4016135953163214`*^9, 
  3.4016139847462945`*^9, 3.4016159010518064`*^9, 3.40161615385532*^9, 
  3.4017100144402127`*^9, 3.4113022948066864`*^9, 3.430477070194525*^9, 
  3.430480948020565*^9}]
}, Open  ]],

Cell["and cumulative distribution function", "Text",
 Editable->False,
 CellChangeTimes->{{3.41241056973987*^9, 3.4124105768801374`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"cdf3layers", " ", "=", " ", 
  RowBox[{"Simplify", "[", 
   RowBox[{"PowerExpand", "[", 
    RowBox[{"Integrate", "[", 
     RowBox[{"pdf3layers", ",", " ", 
      RowBox[{"{", 
       RowBox[{"d", ",", " ", "0", ",", " ", "d"}], "}"}], ",", " ", 
      RowBox[{"Assumptions", "\[Rule]", 
       RowBox[{
        RowBox[{"Re", "[", "\[Alpha]", "]"}], ">", "0"}]}]}], "]"}], "]"}], 
   "]"}]}]], "Input",
 CellChangeTimes->{{3.401452401421875*^9, 3.40145241325*^9}, {
  3.401452521859375*^9, 3.40145252946875*^9}, {3.4014526429375*^9, 
  3.40145265121875*^9}, {3.401526904671296*^9, 3.401526907445285*^9}}],

Cell[BoxData[
 RowBox[{"1", "-", 
  FractionBox[
   SuperscriptBox[
    RowBox[{"Gamma", "[", 
     RowBox[{"\[Alpha]", ",", 
      FractionBox[
       RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}], "]"}], "3"], 
   SuperscriptBox[
    RowBox[{"Gamma", "[", "\[Alpha]", "]"}], "3"]]}]], "Output",
 CellChangeTimes->{3.401526965779165*^9, 3.4015272197643776`*^9, 
  3.4015288808028336`*^9, 3.401529118504632*^9, 3.4016136071533422`*^9, 
  3.4016140424793105`*^9, 3.4016159585344625`*^9, 3.4016162120389843`*^9, 
  3.401710071832739*^9, 3.4113023522893424`*^9, 3.430477126655712*^9, 
  3.430481004291478*^9}]
}, Open  ]],

Cell["For four layers, we obtain", "Text",
 Editable->False,
 CellChangeTimes->{{3.412410589207864*^9, 3.412410600253747*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"pdf4layers", " ", "=", " ", 
  RowBox[{"4", 
   RowBox[{"(", 
    RowBox[{"1", "-", "cdf3layers"}], ")"}], "pdfd"}]}], "\n", 
 RowBox[{"cdf4layers", " ", "=", " ", 
  RowBox[{"Simplify", "[", 
   RowBox[{"PowerExpand", "[", 
    RowBox[{"Integrate", "[", 
     RowBox[{"pdf4layers", ",", " ", 
      RowBox[{"{", 
       RowBox[{"d", ",", " ", "0", ",", " ", "d"}], "}"}], ",", " ", 
      RowBox[{"Assumptions", "\[Rule]", 
       RowBox[{
        RowBox[{"Re", "[", "\[Alpha]", "]"}], ">", "0"}]}]}], "]"}], "]"}], 
   "]"}]}]}], "Input",
 CellChangeTimes->{{3.401526837024024*^9, 3.4015268526164446`*^9}, {
  3.401527322752467*^9, 3.4015273434722605`*^9}}],

Cell[BoxData[
 FractionBox[
  RowBox[{"4", " ", 
   SuperscriptBox["d", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Alpha]"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox[
      RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}]], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox["\[Beta]", "\[Alpha]"], ")"}], 
    RowBox[{"-", "\[Alpha]"}]], " ", 
   SuperscriptBox[
    RowBox[{"Gamma", "[", 
     RowBox[{"\[Alpha]", ",", 
      FractionBox[
       RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}], "]"}], "3"]}], 
  SuperscriptBox[
   RowBox[{"Gamma", "[", "\[Alpha]", "]"}], "4"]]], "Output",
 CellChangeTimes->{{3.40152733889568*^9, 3.401527343592434*^9}, 
   3.4015274107289715`*^9, 3.401528880923006*^9, 3.4015291186248045`*^9, 
   3.4016136119802833`*^9, 3.4016140426395407`*^9, 3.401615958644621*^9, 
   3.4016162121391277`*^9, 3.4017100719729404`*^9, 3.4113023524095154`*^9, 
   3.43047712676587*^9, 3.430481004381608*^9}],

Cell[BoxData[
 RowBox[{"1", "-", 
  FractionBox[
   SuperscriptBox[
    RowBox[{"Gamma", "[", 
     RowBox[{"\[Alpha]", ",", 
      FractionBox[
       RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}], "]"}], "4"], 
   SuperscriptBox[
    RowBox[{"Gamma", "[", "\[Alpha]", "]"}], "4"]]}]], "Output",
 CellChangeTimes->{{3.40152733889568*^9, 3.401527343592434*^9}, 
   3.4015274107289715`*^9, 3.401528880923006*^9, 3.4015291186248045`*^9, 
   3.4016136119802833`*^9, 3.4016140426395407`*^9, 3.401615958644621*^9, 
   3.4016162121391277`*^9, 3.4017100719729404`*^9, 3.4113023524095154`*^9, 
   3.43047712676587*^9, 3.4304810625352287`*^9}]
}, Open  ]],

Cell[TextData[{
 "Our previous outputs suggest the following general form for the probability \
density of the smallest of ",
 StyleBox["n ",
  FontSlant->"Italic"],
 "pores:"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.4124106050206013`*^9, 3.4124106453485904`*^9}, 
   3.4124158088233023`*^9}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pdf", "[", 
   RowBox[{"n_", ",", "d"}], "]"}], ":=", " ", 
  RowBox[{"n", " ", 
   RowBox[{"(", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"1", "-", "cdfd"}], ")"}], 
     RowBox[{"n", "-", "1"}]], ")"}], "pdfd"}]}], "\[IndentingNewLine]", 
 RowBox[{"pdf", "[", 
  RowBox[{"n", ",", "d"}], "]"}]}], "Input",
 CellChangeTimes->{{3.401527705442749*^9, 3.401527790064429*^9}, {
  3.4015279285235233`*^9, 3.401527953599581*^9}, {3.401528028236904*^9, 
  3.4015280566076994`*^9}, {3.4015282020268016`*^9, 3.4015282392603407`*^9}}],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["d", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Alpha]"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox[
      RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}]], " ", "n", " ", 
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox["\[Beta]", "\[Alpha]"], ")"}], 
    RowBox[{"-", "\[Alpha]"}]], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox[
      RowBox[{"Gamma", "[", 
       RowBox[{"\[Alpha]", ",", 
        FractionBox[
         RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}], "]"}], 
      RowBox[{"Gamma", "[", "\[Alpha]", "]"}]], ")"}], 
    RowBox[{
     RowBox[{"-", "1"}], "+", "n"}]]}], 
  RowBox[{"Gamma", "[", "\[Alpha]", "]"}]]], "Output",
 CellChangeTimes->{3.40152824044204*^9, 3.401528944204*^9, 
  3.401529182196216*^9, 3.40161362046248*^9, 3.4016141020950336`*^9, 
  3.4016160179599123`*^9, 3.401616271153987*^9, 3.4017101356344814`*^9, 
  3.411302410112488*^9, 3.4304771863815937`*^9, 3.4304810626554017`*^9}]
}, Open  ]],

Cell["and the cdf is:", "Text",
 Editable->False,
 CellChangeTimes->{{3.4124106517177486`*^9, 3.4124106549624147`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"cdf", "[", 
   RowBox[{"n_", ",", "d"}], "]"}], ":=", 
  RowBox[{"1", "-", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", "cdfd"}], ")"}], "n"]}]}], "\[IndentingNewLine]", 
 RowBox[{"cdf", "[", 
  RowBox[{"n", ",", "d"}], "]"}]}], "Input",
 CellChangeTimes->{{3.401527705442749*^9, 3.401527790064429*^9}, {
  3.4015279285235233`*^9, 3.401527953599581*^9}, {3.401528028236904*^9, 
  3.4015280566076994`*^9}, {3.4015282020268016`*^9, 3.4015282392603407`*^9}}],

Cell[BoxData[
 RowBox[{"1", "-", 
  SuperscriptBox[
   RowBox[{"(", 
    FractionBox[
     RowBox[{"Gamma", "[", 
      RowBox[{"\[Alpha]", ",", 
       FractionBox[
        RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}], "]"}], 
     RowBox[{"Gamma", "[", "\[Alpha]", "]"}]], ")"}], "n"]}]], "Output",
 CellChangeTimes->{3.401528332143901*^9, 3.4015289443442016`*^9, 
  3.401529182236274*^9, 3.4016136205025377`*^9, 3.4016141021451054`*^9, 
  3.401616017979941*^9, 3.401616271174016*^9, 3.401710135644496*^9, 
  3.411302410152546*^9, 3.4304771864817376`*^9, 3.430481062705474*^9}]
}, Open  ]],

Cell["\<\
We render a surface showing the influence of the number of layers on the \
probability density of pore radii in units of the fibre width:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.412410696532189*^9, 3.4124107458430943`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"\[Beta]", " ", "=", " ", 
   RowBox[{"dbar", "/", "\[Omega]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Alpha]", " ", "=", " ", 
   RowBox[{"1", "/", 
    SuperscriptBox["cvd", "2"]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"cvd", " ", "=", " ", 
   RowBox[{
    SqrtBox[
     RowBox[{"16", "-", 
      SuperscriptBox["\[Pi]", "2"]}]], "/", "\[Pi]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Epsilon]", "=", ".5"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"pdf", "[", 
     RowBox[{"n", ",", "d"}], "]"}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"d", ",", " ", "0", ",", " ", "6"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", " ", "1", ",", " ", "10"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", " ", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<d/\[Omega]\>\"", ",", " ", "\"\<n\>\""}], "}"}]}]}], 
  "]"}], "\[IndentingNewLine]", 
 RowBox[{"\[Epsilon]", " ", "=."}]}], "Input",
 CellChangeTimes->{{3.4015287585770817`*^9, 3.401528787298381*^9}, {
  3.4015290375882797`*^9, 3.4015290619132576`*^9}, {3.401529119836547*^9, 
  3.4015292196400576`*^9}, {3.40152925064464*^9, 3.4015293557958403`*^9}, {
  3.4015294950560865`*^9, 3.4015296460031376`*^9}, {3.4015311397710705`*^9, 
  3.4015311851663456`*^9}}],

Cell[BoxData[
 Graphics3DBox[GraphicsComplex3DBox[CompressedData["
1:eJx1nXlYjc8b8CtKIe0pLdpkD2UN53m0KUKRNSqlslQIJUJSlmTpICopkpSU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   "], {{
     {EdgeForm[None], GraphicsGroup3DBox[{Polygon3DBox[CompressedData["
1:eJw1m3ngVsPbxp9zzpypiKRUpIiStURFlpStRbZok6IIRRSVVArZlyxJpCRJ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         "]], Polygon3DBox[CompressedData["
1:eJwtm3fg19Mbxd/rfkpCNKw0EamUyqa0pGUkkhQNqxISyfglUilEVFT2jLS0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         "]], Polygon3DBox[CompressedData["
1:eJwt1wm8jdUexvHtnLOPjBXhY6hrFg6NSJRZSlKiXFOKdEtIqRRSieY5iWQI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         "]], 
        Polygon3DBox[{{1082, 1246, 770, 895, 1511, 1081}}]}]}, {}, {}, {}}, {
     Line3DBox[{381, 1, 349, 228, 1001, 16, 350, 1014, 31, 351, 1028, 46, 353,
       61, 1377, 355, 76, 1378, 357, 91, 1379, 359, 1080, 106, 361, 1107, 121,
       363, 1121, 136, 365, 1135, 151, 1386, 367, 166, 1387, 369, 181, 1388, 
      371, 196, 1335, 270, 420, 211, 373, 269, 419, 667, 212, 434, 271, 668, 
      213, 274, 669, 214, 275, 215, 670, 216, 671, 217, 672, 673, 218, 674, 
      219, 675, 220, 676, 221, 677, 222, 678, 223, 679, 224, 910, 277, 421, 
      225, 380, 303, 1201, 210, 1187, 195, 1173, 180, 1160, 165, 1147, 150, 
      1134, 135, 1120, 120, 1106, 1105, 105, 1078, 90, 1065, 75, 1052, 60, 
      1040, 45, 1027, 30, 1296, 230, 382, 15, 378, 280, 485, 14, 483, 13, 481,
       12, 479, 11, 477, 10, 475, 9, 473, 8, 470, 469, 7, 467, 6, 465, 5, 463,
       4, 461, 3, 459, 2, 989, 435, 278, 381}], {
      Line3DBox[{1001, 1294, 1347, 1361, 680, 1340, 1295, 1522, 1202, 1002, 
       1523, 1203, 1003, 1524, 1204, 1004, 1525, 1205, 1005, 1526, 1206, 1006,
        1527, 1433, 1623, 1007, 1528, 1207, 1008, 1529, 1208, 1009, 1530, 
       1209, 1010, 1531, 1210, 1011, 1532, 1211, 1012, 1533, 1212, 1013, 1509,
        1534, 1213, 1296}], 
      Line3DBox[{1014, 1297, 1348, 318, 1721, 1341, 1298, 1389, 1726, 1214, 
       1015, 1535, 1215, 1016, 1536, 1216, 1017, 1537, 1217, 1018, 1538, 1218,
        1019, 1539, 1434, 1624, 1020, 1435, 1625, 1021, 1540, 1219, 1022, 
       1541, 1220, 1023, 1542, 1221, 1024, 1543, 1222, 1025, 1544, 1223, 1026,
        1545, 1224, 1027}], 
      Line3DBox[{1028, 1376, 1299, 1349, 1362, 936, 1342, 1424, 1300, 1390, 
       1436, 1708, 1343, 1301, 711, 1029, 1546, 1225, 1030, 1547, 1226, 1031, 
       1548, 1227, 1032, 1549, 1437, 1626, 1033, 1438, 1627, 1034, 1439, 1628,
        1035, 1550, 1228, 1036, 1551, 1229, 1037, 1552, 1230, 1038, 1553, 
       1231, 1039, 1554, 1232, 1040}], 
      Line3DBox[{353, 236, 915, 284, 237, 916, 285, 238, 1709, 1041, 727, 
       1042, 1555, 1233, 1043, 1556, 1234, 1044, 1557, 1440, 1629, 1045, 1441,
        1630, 1046, 1442, 1631, 1047, 1443, 1632, 1048, 735, 1049, 1558, 1235,
        1050, 1559, 1236, 1051, 1560, 1237, 1052}], 
      Line3DBox[{1065, 1242, 1566, 1064, 1241, 1565, 1063, 1240, 1564, 1062, 
       1638, 1449, 1061, 1637, 1448, 1060, 1636, 1447, 1059, 1635, 1446, 1058,
        1634, 1445, 1563, 1057, 1239, 1562, 1056, 1238, 1561, 1055, 1633, 
       1444, 1054, 892, 1306, 1053, 1716, 1305, 1391, 1304, 1392, 1393, 1519, 
       1723, 1303, 1350, 1302, 1416, 1377}], 
      Line3DBox[{1078, 1244, 1569, 1077, 767, 1076, 1647, 1457, 1075, 1646, 
       1456, 1074, 1645, 1455, 1073, 1644, 1454, 1072, 1643, 1453, 1071, 1642,
        1452, 1568, 1070, 1243, 1567, 1069, 1641, 1451, 1068, 1640, 1450, 
       1067, 1510, 1639, 1310, 1066, 893, 1394, 1309, 1395, 1396, 1717, 1368, 
       1308, 1351, 1307, 1417, 1378}], 
      Line3DBox[{1080, 1380, 1381, 1353, 1363, 1720, 1245, 1427, 1425, 1426, 
       1400, 1727, 1246, 1082, 1570, 1247, 1084, 1571, 1248, 1086, 1572, 1249,
        1088, 1573, 1250, 1090, 1574, 776, 1092, 1575, 1251, 1094, 1576, 1252,
        1096, 1577, 1253, 1098, 1578, 1254, 1100, 1579, 1255, 1102, 1580, 
       1256, 1104, 1581, 1257, 1106}], 
      Line3DBox[{1105, 783, 1103, 1659, 1467, 1101, 1658, 1466, 1099, 1657, 
       1465, 1097, 1656, 1464, 1095, 1655, 1463, 1093, 1654, 1462, 1091, 1653,
        775, 1089, 1652, 1461, 1087, 1651, 1460, 1085, 1650, 1459, 1083, 1649,
        1458, 1081, 1511, 1648, 1397, 1399, 1398, 1079, 957, 1369, 1352, 1419,
        1418, 1379}], 
      Line3DBox[{1107, 1382, 1383, 1354, 1364, 1468, 1722, 1430, 1428, 1429, 
       1401, 1728, 1258, 1108, 1582, 1259, 1109, 1583, 1260, 1110, 1584, 1261,
        1111, 1585, 1262, 1112, 1586, 1469, 1660, 1113, 792, 1114, 1587, 1263,
        1115, 1588, 1264, 1116, 1589, 1265, 1117, 1590, 1266, 1118, 1591, 
       1267, 1119, 1592, 1268, 1120}], 
      Line3DBox[{1121, 1384, 1311, 1355, 1365, 1366, 1730, 1344, 1431, 1312, 
       1402, 1470, 1710, 1122, 1593, 1269, 1123, 1594, 1270, 1124, 1595, 1271,
        1125, 1596, 1272, 1126, 1597, 1471, 1661, 1127, 1472, 1662, 1128, 808,
        1129, 1598, 1273, 1130, 1599, 1274, 1131, 1600, 1275, 1132, 1601, 
       1276, 1133, 1602, 1277, 1134}], 
      Line3DBox[{1135, 1385, 1313, 1356, 1367, 1711, 1518, 1345, 1432, 1314, 
       1403, 1473, 1712, 1136, 1315, 1474, 1713, 1346, 1316, 1603, 1278, 1137,
        1604, 1279, 1138, 1605, 1280, 1139, 1606, 1475, 1663, 1140, 1476, 
       1664, 1141, 1477, 1665, 1142, 1478, 1666, 1143, 1607, 1281, 1144, 1608,
        1282, 1145, 1609, 1283, 1146, 1610, 1284, 1147}], 
      Line3DBox[{1160, 1289, 1616, 1159, 1288, 1615, 1158, 1287, 1614, 1157, 
       1672, 1483, 1156, 1671, 1482, 1155, 1670, 1481, 1154, 1669, 1480, 1153,
        1668, 1479, 1613, 1152, 1286, 1612, 1151, 1285, 1611, 1150, 1324, 
       1667, 1323, 1149, 1718, 1322, 1321, 1148, 1714, 1320, 1404, 1319, 1405,
        1406, 1520, 1724, 1318, 1357, 1317, 1420, 1386}], 
      Line3DBox[{1173, 1292, 1620, 1172, 1291, 1619, 1171, 1681, 1490, 1170, 
       1680, 1489, 1169, 1679, 1488, 1168, 1678, 1487, 1167, 1677, 1486, 1166,
        1676, 1485, 1618, 1165, 1290, 1617, 1164, 1675, 1484, 1163, 1331, 
       1674, 1330, 1162, 1512, 1673, 1329, 1161, 1715, 1328, 1407, 1327, 1408,
        1409, 1719, 1370, 1326, 1358, 1325, 1421, 1387}], 
      Line3DBox[{1187, 1293, 1622, 1186, 1693, 1499, 1185, 1692, 1498, 1184, 
       1691, 1497, 1183, 1690, 1496, 1182, 1689, 1495, 1181, 1688, 1494, 1180,
        1687, 1493, 1621, 1179, 1686, 1492, 1178, 1685, 1491, 1177, 1334, 
       1684, 1333, 1176, 1514, 1683, 1332, 1175, 1513, 1682, 1410, 1412, 1411,
        1174, 1729, 1371, 1372, 1359, 1423, 1422, 1388}], 
      Line3DBox[{1201, 1339, 1707, 1517, 1200, 1706, 1508, 1199, 1705, 1507, 
       1198, 1704, 1506, 1197, 1703, 1505, 1196, 1702, 1504, 1195, 1701, 1503,
        1194, 1700, 1699, 1502, 1193, 1698, 1501, 1192, 1697, 1500, 1191, 
       1338, 1696, 1337, 1190, 1516, 1695, 1336, 1189, 1515, 1694, 1413, 1415,
        1414, 1188, 1521, 1725, 1375, 1360, 1374, 1373, 1335}]}, {
      Line3DBox[{459, 883, 681, 1522, 460, 886, 695, 1726, 488, 887, 914, 
       1708, 710, 503, 889, 916, 725, 518, 891, 1716, 918, 740, 533, 893, 920,
        755, 548, 1648, 895, 770, 1727, 563, 896, 785, 1728, 578, 897, 1710, 
       800, 593, 898, 1712, 815, 608, 900, 1714, 829, 623, 902, 1715, 843, 
       638, 1682, 904, 857, 653, 1694, 906, 871, 668}], 
      Line3DBox[{461, 682, 1523, 462, 696, 1535, 489, 888, 711, 504, 890, 
       1709, 726, 519, 892, 741, 534, 1639, 894, 756, 549, 1649, 771, 1570, 
       564, 786, 1582, 579, 801, 1593, 594, 899, 925, 1713, 816, 609, 901, 
       1718, 927, 830, 624, 1673, 903, 929, 844, 639, 1683, 905, 931, 858, 
       654, 1695, 907, 933, 872, 669}], 
      Line3DBox[{463, 683, 1524, 464, 697, 1536, 490, 712, 1546, 505, 727, 
       520, 1633, 742, 535, 1640, 757, 550, 1650, 772, 1571, 565, 787, 1583, 
       580, 802, 1594, 595, 254, 1603, 610, 1667, 258, 625, 1674, 262, 640, 
       1684, 266, 655, 1696, 273, 275}], 
      Line3DBox[{465, 684, 1525, 466, 698, 1537, 491, 713, 1547, 506, 728, 
       1555, 521, 743, 1561, 536, 1641, 758, 551, 1651, 773, 1572, 566, 788, 
       1584, 581, 803, 1595, 596, 817, 1604, 611, 831, 1611, 626, 1675, 845, 
       641, 1685, 859, 656, 1697, 873, 670}], 
      Line3DBox[{467, 685, 1526, 468, 699, 1538, 492, 714, 1548, 507, 729, 
       1556, 522, 744, 1562, 537, 759, 1567, 552, 1652, 774, 1573, 567, 789, 
       1585, 582, 804, 1596, 597, 818, 1605, 612, 832, 1612, 627, 846, 1617, 
       642, 1686, 860, 657, 1698, 874, 671}], 
      Line3DBox[{469, 686, 1527, 471, 700, 1539, 493, 715, 1549, 508, 730, 
       1557, 523, 745, 1563, 538, 760, 1568, 553, 775, 1574, 568, 790, 1586, 
       583, 805, 1597, 598, 819, 1606, 613, 833, 1613, 628, 847, 1618, 643, 
       861, 1621, 658, 1699, 875, 672}], 
      Line3DBox[{473, 688, 1528, 474, 1625, 702, 495, 1627, 717, 510, 1630, 
       732, 525, 1635, 747, 540, 1643, 762, 555, 1654, 777, 1575, 570, 792, 
       585, 1662, 807, 600, 1664, 821, 615, 1669, 835, 630, 1677, 849, 645, 
       1688, 863, 660, 1701, 877, 674}], 
      Line3DBox[{475, 689, 1529, 476, 703, 1540, 496, 1628, 718, 511, 1631, 
       733, 526, 1636, 748, 541, 1644, 763, 556, 1655, 778, 1576, 571, 793, 
       1587, 586, 808, 601, 1665, 822, 616, 1670, 836, 631, 1678, 850, 646, 
       1689, 864, 661, 1702, 878, 675}], 
      Line3DBox[{477, 690, 1530, 478, 704, 1541, 497, 719, 1550, 512, 1632, 
       734, 527, 1637, 749, 542, 1645, 764, 557, 1656, 779, 1577, 572, 794, 
       1588, 587, 809, 1598, 602, 1666, 823, 617, 1671, 837, 632, 1679, 851, 
       647, 1690, 865, 662, 1703, 879, 676}], 
      Line3DBox[{479, 691, 1531, 480, 705, 1542, 498, 720, 1551, 513, 735, 
       528, 1638, 750, 543, 1646, 765, 558, 1657, 780, 1578, 573, 795, 1589, 
       588, 810, 1599, 603, 824, 1607, 618, 1672, 838, 633, 1680, 852, 648, 
       1691, 866, 663, 1704, 880, 677}], 
      Line3DBox[{481, 692, 1532, 482, 706, 1543, 499, 721, 1552, 514, 736, 
       1558, 529, 751, 1564, 544, 1647, 766, 559, 1658, 781, 1579, 574, 796, 
       1590, 589, 811, 1600, 604, 825, 1608, 619, 839, 1614, 634, 1681, 853, 
       649, 1692, 867, 664, 1705, 881, 678}], 
      Line3DBox[{483, 693, 1533, 484, 707, 1544, 500, 722, 1553, 515, 737, 
       1559, 530, 752, 1565, 545, 767, 560, 1659, 782, 1580, 575, 797, 1591, 
       590, 812, 1601, 605, 826, 1609, 620, 840, 1615, 635, 854, 1619, 650, 
       1693, 868, 665, 1706, 882, 679}], 
      Line3DBox[{485, 884, 885, 1534, 486, 708, 1545, 501, 723, 1554, 516, 
       738, 1560, 531, 753, 1566, 546, 768, 1569, 561, 783, 1581, 576, 798, 
       1592, 591, 813, 1602, 606, 827, 1610, 621, 841, 1616, 636, 855, 1620, 
       651, 869, 1622, 666, 1707, 908, 909, 910}], 
      Line3DBox[{667, 870, 976, 973, 932, 972, 988, 971, 1725, 652, 856, 1000,
        946, 930, 970, 987, 1729, 969, 637, 842, 999, 945, 928, 968, 1719, 
       986, 967, 622, 828, 998, 944, 926, 1724, 966, 985, 965, 607, 814, 997, 
       943, 1711, 924, 964, 984, 963, 592, 799, 996, 1730, 942, 923, 962, 983,
        961, 577, 784, 1722, 995, 941, 922, 960, 982, 959, 562, 1720, 769, 
       994, 940, 921, 958, 981, 957, 547, 754, 993, 939, 919, 956, 1717, 980, 
       955, 532, 739, 992, 938, 917, 1723, 954, 979, 953, 517, 724, 991, 937, 
       915, 952, 978, 951, 502, 709, 990, 936, 913, 950, 977, 949, 487, 694, 
       1721, 935, 912, 948, 947, 458, 680, 934, 911, 975, 974, 989}], 
      Line3DBox[{673, 876, 1700, 659, 862, 1687, 644, 848, 1676, 629, 834, 
       1668, 614, 820, 1663, 599, 806, 1661, 584, 791, 1660, 569, 776, 1653, 
       554, 761, 1642, 539, 746, 1634, 524, 731, 1629, 509, 716, 1626, 494, 
       701, 1624, 472, 687, 1623, 470}]}, {}, {}}},
   VertexNormals->CompressedData["
1:eJx0m3c01//7/1VWCg2zYVSEjCSV8LpKKYSsbKWIsiqEitCQWSQrsveekfAw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    "]],
  Axes->True,
  AxesLabel->{
    FormBox["\"d/\[Omega]\"", TraditionalForm], 
    FormBox["\"n\"", TraditionalForm]},
  BoxRatios->{1, 1, 0.4},
  Method->{"RotationControl" -> "Globe"},
  PlotRange->{All, All, All},
  PlotRangePadding->{Automatic, Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.4015311873494844`*^9, 3.4016136221749425`*^9, 
  3.401614103456992*^9, 3.401616019211712*^9, 3.40161627243583*^9, 
  3.4017101374871454`*^9, 3.4113024121053534`*^9, 3.430477191669197*^9, 
  3.4304810643278065`*^9},
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell[TextData[{
 "We determine the mean, variance and coefficient of variation using \
numerical integration. Our example assumes ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    OverscriptBox["c", "_"], "=", " ", "5"}], TraditionalForm]]],
 " and \[Epsilon] = 0.5:"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.4124107676444435`*^9, 3.412410813850885*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"Clear", "[", "\[Beta]", "]"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"n", " ", "=", " ", 
   RowBox[{"cbar", "/", 
    RowBox[{"Log", "[", 
     RowBox[{"1", "/", "\[Epsilon]"}], "]"}]}]}], ";", 
  RowBox[{"\[Beta]", " ", "=", " ", 
   RowBox[{"dbar", "/", "\[Omega]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Epsilon]", " ", "=", " ", "0.5"}], ";", 
  RowBox[{"cbar", " ", "=", " ", "5"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"dbarmp", " ", "=", " ", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"d", " ", 
     RowBox[{"pdf", "[", 
      RowBox[{"n", ",", "d"}], "]"}]}], " ", ",", " ", 
    RowBox[{"{", 
     RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], 
   "]"}]}], "\[IndentingNewLine]", 
 RowBox[{"dvarmp", " ", "=", " ", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"d", "-", "dbarmp"}], ")"}], "2"], " ", 
     RowBox[{"pdf", "[", 
      RowBox[{"n", ",", "d"}], "]"}]}], " ", ",", " ", 
    RowBox[{"{", 
     RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], 
   "]"}]}], "\[IndentingNewLine]", 
 RowBox[{"cvdmp", " ", "=", " ", 
  RowBox[{
   SqrtBox["dvarmp"], "/", "dbarmp"}]}]}], "Input",
 CellChangeTimes->{{3.401534510137419*^9, 3.4015345659576845`*^9}, {
  3.401537385722307*^9, 3.401537390368989*^9}}],

Cell[BoxData["0.6753651329736996`"], "Output",
 CellChangeTimes->{{3.4015345203921647`*^9, 3.4015345290946784`*^9}, 
   3.401534566718779*^9, 3.4015374937776833`*^9, 3.4016136234968433`*^9, 
   3.401614103937683*^9, 3.4016160195922594`*^9, 3.4016162729665937`*^9, 
   3.4017101383784275`*^9, 3.4113024129265347`*^9, 3.430480350421259*^9, 
   3.4304810657097936`*^9}],

Cell[BoxData["0.22401468042145192`"], "Output",
 CellChangeTimes->{{3.4015345203921647`*^9, 3.4015345290946784`*^9}, 
   3.401534566718779*^9, 3.4015374937776833`*^9, 3.4016136234968433`*^9, 
   3.401614103937683*^9, 3.4016160195922594`*^9, 3.4016162729665937`*^9, 
   3.4017101383784275`*^9, 3.4113024129265347`*^9, 3.430480350421259*^9, 
   3.4304810658700237`*^9}],

Cell[BoxData["0.700808893310335`"], "Output",
 CellChangeTimes->{{3.4015345203921647`*^9, 3.4015345290946784`*^9}, 
   3.401534566718779*^9, 3.4015374937776833`*^9, 3.4016136234968433`*^9, 
   3.401614103937683*^9, 3.4016160195922594`*^9, 3.4016162729665937`*^9, 
   3.4017101383784275`*^9, 3.4113024129265347`*^9, 3.430480350421259*^9, 
   3.430481065920096*^9}]
}, Open  ]],

Cell[TextData[{
 "To investigate the influence of mean coverage on the mean and coefficient \
of variation of pore radii, we perform multiple integrations within a ",
 StyleBox["Table", "Input"],
 " environment:"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.412410835111456*^9, 3.412410883711339*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"Clear", "[", 
  RowBox[{"cbar", ",", "n", ",", "\[Beta]"}], "]"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"n", " ", "=", " ", 
   RowBox[{"cbar", "/", 
    RowBox[{"Log", "[", 
     RowBox[{"1", "/", "\[Epsilon]"}], "]"}]}]}], ";", 
  RowBox[{"\[Beta]", " ", "=", " ", 
   RowBox[{"dbar", "/", "\[Omega]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Epsilon]", " ", "=", " ", "0.5"}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"dbartab", " ", "=", " ", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"d", " ", 
        RowBox[{"pdf", "[", 
         RowBox[{"n", ",", "d"}], "]"}]}], " ", ",", " ", 
       RowBox[{"{", 
        RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], "]"}],
      ",", " ", 
     RowBox[{"{", 
      RowBox[{"cbar", ",", " ", "1", ",", " ", "50"}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"dvartab", " ", "=", " ", 
   RowBox[{
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{
         SuperscriptBox["d", "2"], " ", 
         RowBox[{"pdf", "[", 
          RowBox[{"n", ",", "d"}], "]"}]}], " ", ",", " ", 
        RowBox[{"{", 
         RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], 
       "]"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"cbar", ",", " ", "1", ",", " ", "50"}], "}"}]}], "]"}], " ", 
    "-", " ", 
    SuperscriptBox["dbartab", "2"]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"cvdtab", " ", "=", " ", 
   RowBox[{
    SqrtBox["dvartab"], "/", "dbartab"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"GraphicsArray", "[", 
  RowBox[{"{", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"ListPlot", "[", 
      RowBox[{"dbartab", ",", " ", 
       RowBox[{"AxesLabel", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
         "\"\<\!\(\*OverscriptBox[\(c\), \(_\)]\)\>\"", ",", 
          "\"\<\!\(\*OverscriptBox[\(d\), \(_\)]\)/\[Omega]\>\""}], "}"}]}], 
       ",", 
       RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}], ",", " ", 
     RowBox[{"ListPlot", "[", 
      RowBox[{"cvdtab", ",", " ", 
       RowBox[{"AxesLabel", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
         "\"\<\!\(\*OverscriptBox[\(c\), \(_\)]\)\>\"", ",", 
          "\"\<cv(d)\>\""}], "}"}]}], ",", 
       RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], "}"}], "}"}], 
  "]"}]}], "Input",
 CellChangeTimes->{{3.401612885685923*^9, 3.4016131325709267`*^9}, {
  3.4016131647171507`*^9, 3.4016131703252144`*^9}, {3.4016132010694227`*^9, 
  3.4016133844030433`*^9}, {3.4016134367883697`*^9, 3.401613471688554*^9}, {
  3.401613784368165*^9, 3.4016137869618945`*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {InsetBox[
     GraphicsBox[{{}, 
       {Hue[0.67, 0.6, 0.6], 
        PointBox[{{1., 2.1721149805851665`}, {2., 1.2945948309283404`}, {3., 
         0.9666307133355421}, {4., 0.7889339086908291}, {5., 
         0.6753651329736996}, {6., 0.5955766395591386}, {7., 
         0.5359615560025103}, {8., 0.4894486024771947}, {9., 
         0.4519736352398693}, {10., 0.4210228955456701}, {11., 
         0.3949519678105518}, {12., 0.3726364844787137}, {13., 
         0.35327962259828644`}, {14., 0.33629970225913053`}, {15., 
         0.3212613849768609}, {16., 0.3078318581389581}, {17., 
         0.2957519790578682}, {18., 0.28481671421991933`}, {19., 
         0.2748615422935298}, {20., 0.2657527919724139}, {21., 
         0.2573806406779042}, {22., 0.24965395234750168`}, {23., 
         0.24249641181093493`}, {24., 0.2358435866585108}, {25., 
         0.22964067086972728`}, {26., 0.2238407234480579}, {27., 
         0.2184032931774811}, {28., 0.21329330651717915`}, {29., 
         0.20848020023543626`}, {30., 0.2039372017700021}, {31., 
         0.19964075392212174`}, {32., 0.19557003719939653`}, {33., 
         0.1917065776420843}, {34., 0.18803392046127604`}, {35., 
         0.1845373572195541}, {36., 0.1812036965119198}, {37., 
         0.17802107021138786`}, {38., 0.17497876896373454`}, {39., 
         0.17206710187204463`}, {40., 0.16927727629237932`}, {41., 
         0.16660129443340943`}, {42., 0.16403186406289477`}, {43., 
         0.16156232111013039`}, {44., 0.1591865623427141}, {45., 
         0.15689898660956464`}, {46., 0.15469444339705055`}, {47., 
         0.1525681876445135}, {48., 0.15051583992882198`}, {49., 
         0.14853335142544644`}, {50., 0.14661697260417417`}}]}, {}},
      AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
      Axes->True,
      AxesLabel->{
        FormBox[
        "\"\\!\\(\\*OverscriptBox[\\(c\\), \\(_\\)]\\)\"", TraditionalForm], 
        FormBox[
        "\"\\!\\(\\*OverscriptBox[\\(d\\), \\(_\\)]\\)/\[Omega]\"", 
         TraditionalForm]},
      AxesOrigin->{0, 0.5},
      PlotRange->{All, All},
      PlotRangeClipping->True,
      PlotRangePadding->{Automatic, Automatic}], {192., -116.80842387373012`},
      ImageScaled[{0.5, 0.5}], {360., 222.49223594996212`}], InsetBox[
     GraphicsBox[{{}, 
       {Hue[0.67, 0.6, 0.6], 
        PointBox[{{1., 0.7684599061269487}, {2., 0.7352353900821788}, {3., 
         0.718632885343247}, {4., 0.7081768338423208}, {5., 
         0.7008088945238947}, {6., 0.6952547375879445}, {7., 
         0.6908740163197277}, {8., 0.6873044480428234}, {9., 
         0.6843234133424593}, {10., 0.6817854861034237}, {11., 
         0.6795910678733953}, {12., 0.6776693494518844}, {13., 
         0.6759683954569197}, {14., 0.6744491386256638}, {15., 
         0.6730815548269081}, {16., 0.6718421222207864}, {17., 
         0.670712129161992}, {18., 0.6696764728577624}, {19., 
         0.6687228122493277}, {20., 0.6678409509530278}, {21., 
         0.6670223807923001}, {22., 0.6662599387408816}, {23., 
         0.6655475414830118}, {24., 0.6648799995042124}, {25., 
         0.6642528340964502}, {26., 0.6636621864967176}, {27., 
         0.6631046418819095}, {28., 0.6625772812597508}, {29., 
         0.6620774892206817}, {30., 0.6616029797753856}, {31., 
         0.6611517088176088}, {32., 0.6607218599669427}, {33., 
         0.6603118089505472}, {34., 0.6599200982769375}, {35., 
         0.6595454158717077}, {36., 0.6591865769647361}, {37., 
         0.6588425086588084}, {38., 0.6585122367233406}, {39., 
         0.6581948742432511}, {40., 0.6578896096888427}, {41., 
         0.6575957091060438}, {42., 0.6573124849974502}, {43., 
         0.6570393206548717}, {44., 0.6567756425688358}, {45., 
         0.6565209163055719}, {46., 0.6562746553761001}, {47., 
         0.6560364072432214}, {48., 0.655805752735662}, {49., 
         0.6555823014358644}, {50., 0.6553656915288412}}]}, {}},
      AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
      Axes->True,
      AxesLabel->{
        FormBox[
        "\"\\!\\(\\*OverscriptBox[\\(c\\), \\(_\\)]\\)\"", TraditionalForm], 
        FormBox["\"cv(d)\"", TraditionalForm]},
      AxesOrigin->{0, 0.66},
      PlotRange->{All, All},
      PlotRangeClipping->True,
      PlotRangePadding->{Automatic, Automatic}], {576., -116.80842387373012`},
      ImageScaled[{0.5, 0.5}], {360., 222.49223594996212`}]}, {}},
  ContentSelectable->True,
  PlotRangePadding->{6, 5}]], "Output",
 CellChangeTimes->{
  3.4016129734120674`*^9, 3.4016130054381185`*^9, {3.4016130443440623`*^9, 
   3.4016130989225426`*^9}, 3.401613171206482*^9, 3.4016132477966127`*^9, {
   3.401613332648624*^9, 3.4016133856147857`*^9}, 3.401613441024461*^9, 
   3.40161347425224*^9, 3.4016137898961134`*^9, 3.4016141061508656`*^9, 
   3.401616021695283*^9, 3.4016162751096754`*^9, 3.4017101406416817`*^9, 
   3.4113024158907967`*^9, 3.430480355869093*^9, 3.430481068393653*^9}]
}, Open  ]],

Cell[TextData[{
 "We compute the influence of fibre width on the mean pore diameter in a \
similar way. Our example assumes a network with mean areal density 20 g ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      SuperscriptBox[
       StyleBox["m",
        FontSlant->"Plain"], 
       RowBox[{"-", "2"}]], " ", "and", " ", "porosity", " ", "\[Epsilon]"}], 
     " ", "=", " ", "0.5"}], ",", "  ", 
    RowBox[{"formed", " ", "from", " ", "nylon", " ", "fibres"}]}], 
   TraditionalForm]]],
 "with circular cross section and density \[Rho] = 1100 kg ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox[
     StyleBox["m",
      FontSlant->"Plain"], 
     RowBox[{"-", "3"}]], ".", " "}], TraditionalForm]]],
 " Note that the code allows the linear density of fibres (\[Delta]) and \
number \nof layers required to model the structure to vary as we vary fibre \
width 1 \[LessEqual] \[Omega] \[LessEqual] 20 \[Mu]m. Note also the careful \
specification of units to give the mean pore radius in \[Mu]m."
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.412410954983824*^9, 3.412411230349781*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"Clear", "[", 
  RowBox[{"cbar", ",", "n", ",", "\[Beta]", ",", "\[Epsilon]"}], 
  "]"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"n", " ", "=", " ", 
   RowBox[{"cbar", "/", 
    RowBox[{"Log", "[", 
     RowBox[{"1", "/", "\[Epsilon]"}], "]"}]}]}], ";", 
  RowBox[{"\[Beta]", " ", "=", " ", "dbar"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Beta]bar", " ", "=", " ", "20"}], " ", 
  RowBox[{"(*", " ", 
   RowBox[{"g", " ", 
    SuperscriptBox["m", 
     RowBox[{"-", "2"}]]}], "*)"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Rho]", " ", "=", " ", 
   RowBox[{"1.1", " ", 
    SuperscriptBox["10", "6"]}]}], " ", 
  RowBox[{"(*", " ", 
   RowBox[{"g", " ", 
    SuperscriptBox["m", 
     RowBox[{"-", "3"}]]}], "*)"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Delta]", "=", " ", 
   RowBox[{"\[Rho]", " ", "\[Pi]", " ", 
    RowBox[{
     SuperscriptBox["\[Omega]", "2"], "/", "4"}]}]}], " ", 
  RowBox[{"(*", " ", 
   RowBox[{"g", " ", 
    SuperscriptBox["m", 
     RowBox[{"-", "1"}]]}], " ", "*)"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"cbar", " ", "=", " ", 
   RowBox[{
    SuperscriptBox["10", 
     RowBox[{"6", " "}]], "\[Beta]bar", " ", 
    RowBox[{"\[Omega]", "/", "\[Delta]"}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Epsilon]", " ", "=", " ", "0.5"}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Omega]dmp", " ", "=", " ", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\[Omega]", ",", 
       RowBox[{"NIntegrate", "[", 
        RowBox[{
         RowBox[{"d", " ", 
          RowBox[{"pdf", "[", 
           RowBox[{"n", ",", "d"}], "]"}]}], ",", " ", 
         RowBox[{"{", 
          RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], 
        "]"}]}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"\[Omega]", ",", " ", "1", ",", " ", "20"}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"\[Omega]dmp", ",", " ", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
     "\"\<\[Omega] (\[Mu]m)\>\"", ",", " ", 
      "\"\<\!\(\*OverscriptBox[\(d\), \(_\)]\) (\[Mu]m)\>\""}], "}"}]}]}], 
  "]"}]}], "Input",
 CellChangeTimes->{{3.401612876913309*^9, 3.4016128806787233`*^9}, {
   3.4016137551060877`*^9, 3.4016137623464994`*^9}, {3.4016139610121665`*^9, 
   3.401614115364114*^9}, {3.4016141587665234`*^9, 3.40161417259641*^9}, {
   3.4016143279898543`*^9, 3.401614330583584*^9}, {3.4016144587578897`*^9, 
   3.401614523200554*^9}, {3.4016146001612177`*^9, 3.40161460838304*^9}, {
   3.4016147066543474`*^9, 3.4016147078160176`*^9}, {3.4016149001425695`*^9, 
   3.4016149825911245`*^9}, {3.401615019193757*^9, 3.4016150376202526`*^9}, {
   3.4016150684846334`*^9, 3.401615085449027*^9}, {3.4016151298528767`*^9, 
   3.4016151397771473`*^9}, {3.401615224569072*^9, 3.401615253630861*^9}, {
   3.4016153006684976`*^9, 3.401615360694811*^9}, {3.4016153922401714`*^9, 
   3.4016153931715107`*^9}, 3.401616079768789*^9}],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], 
    PointBox[{{1., 0.24146904751017173`}, {2., 0.7634416011118068}, {3., 
     1.5047190634494616`}, {4., 2.4420882748820283`}, {5., 
     3.5618962415773536`}, {6., 4.8548535523357925`}, {7., 
     6.3141131275483415`}, {8., 7.934361735900559}, {9., 9.711324943908895}, {
     10., 11.641470643156762`}, {11., 13.721819005884061`}, {12., 
     15.949814218578926`}, {13., 18.323234328188583`}, {14., 
     20.840125765777533`}, {15., 23.4987544876053}, {16., 
     26.29756867705981}, {17., 29.235169713940287`}, {18., 
     32.31028919678684}, {19., 35.52177048839072}, {20., 
     38.86855370216838}}]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox["\"\[Omega] (\[Mu]m)\"", TraditionalForm], 
    FormBox[
    "\"\\!\\(\\*OverscriptBox[\\(d\\), \\(_\\)]\\) (\[Mu]m)\"", 
     TraditionalForm]},
  AxesOrigin->{0, 0},
  PlotRange->{{0., 20.}, {0., 38.86855370216838}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{
  3.4016151413193645`*^9, {3.4016152315991807`*^9, 3.4016152543518977`*^9}, 
   3.4016153611054015`*^9, {3.4016153955248947`*^9, 3.4016154093447666`*^9}, 
   3.401616022366248*^9, 3.4016162758006687`*^9, 3.401710141613078*^9, 
   3.4113024166318626`*^9, 3.4304803615072002`*^9, 3.430481069124704*^9},
 ImageCache->GraphicsData["CompressedBitmap", "\<\
eJztWs1PE0EUn7JdIEo96kEO3EzKX4Af8aMCNZHEixQ90RAJHBSDQLzhQW96
wsSYePHzStBEj8V40RMePYnxD+DqqamdmbI7b+fte7Ow7TYNBKfje7/39s3v
/bqzLTNVXVm4c7e6sjhXHZlcrt5fWJx7MDKxtNw0eTkhcgtCiD8jQs4bzWlr
aDR/G4tykBZ8XpEvnoRPBrM+OeuTs6uBTeQ1PvwpSUMe4osadFu96LiGLkOM
qzEPbPuxEegVxOYhtstq9AOb+YrlPdPR6q64p+xsYaWIjalqJuhuOWktELqf
ZuIA14+4JpNW4kcrUf9X8hb90TQOrboFV6NmHnzbqFk+mJUQXNF4P1YC6xBy
2ZyBnEb4K8hZTs5yeqYcJzSyAdYux3NyUPT1I8mGRFBmYOcynqWp9B16Htrw
xrovz6mYwTg3bL9jLYPyRQHOy2EA6SEmJ4+4hkb6QVkWT06EtNS0W7n5Qdr/
jZdeabXguCNbxNYervD7kU+ooY9Qg5dUDeUf3/9K++yXz7+O1JChGk5qm7Cu
4dTN09r28P27ncbFCy9fPHv6TYwiuLKj7RpiO4VUeByx8WWqW1CzTHULWkNw
22wKqdva0r1PSrfGrR6NKSLsKM8oGwNKdYyxayNi1Dhj+JXbskaixuJWNMzG
2CviY+wVETFc7Zy/HLe2Ahtjr42PsddmxuSDblTat+IKsuI6CweL5eFgnQT8
sItJ7FfueYSCvfgk8wgFPBxQQMA7TkFC/ypC1m6U0VWEol02J6CIgGdNgekP
n5HD5yMfiXmM0PZTp6CevvGHdywz4DrVzKAtB8zsB7ey6VRKM1uQYqqwh+HH
Y6ybWStwA1HTVxYOJGLCFXoD6TaRM2sK2u1/g1D8kYUDik242stfIxQTObOm
oN3+LYTitywcUGzCFXoLoZjImTUFHfc7PVNuI415zsJBY3g4aBEBT5MC4+tE
vUUIbt+2docE3/R3cOPKR2LIPNZmnCiaoncA8JIGR6478g4i2SfxothBJMvD
gWQJeOrvWqfP5d3u/420aI2FI9/XUHDQIgJ+sBtj1hR2nV+595DGzscn2UMa
y8NBYwl411HUE8KrIy0myq0jLW7B7T/h0zf2OtL+Vqq0tq208oR7Vcwf7dSF
2vlROLWNPGGTspZnGzfUAqL8slsMkL9jDNA5F9NNjPbEbc7JP4woYswtBijC
MQYowozpiYfCHtbhKKKTolsM0IljDNAJF5MSo2rOnO1zmas/5cYeO1LjlEYe
i/jlTyEu5rqG+Ma2BQKxI0bmxexNGTuolPQEEVGptU2LVM4JUVf0QaL9OXUG
hD0RRFwuTHCoUyZks20/VhUnmvDMHJYRO3tgKscUt/Au1Wq1R+vr6/Kf8PR4
Y3NzE4bOIqG5/5meTcM=\
\>"]]
}, Open  ]],

Cell[TextData[{
 "By selecting a random porosity and specifying ",
 StyleBox["d",
  FontSlant->"Italic"],
 " in units of fibre width, we show that the probability density of pore \
diameters is well approximated by a gamma distribution with the same mean and \
variance:"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.412411257769208*^9, 3.4124113241446514`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"Clear", "[", 
  RowBox[{
  "cbar", ",", "n", ",", "\[Beta]", ",", "\[Epsilon]", ",", "\[Beta]bar", ",",
    "\[Rho]", ",", "\[Omega]"}], "]"}], "\[IndentingNewLine]", 
 RowBox[{"SeedRandom", "[", "1", "]"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"n", " ", "=", " ", 
   RowBox[{"cbar", "/", 
    RowBox[{"Log", "[", 
     RowBox[{"1", "/", "\[Epsilon]"}], "]"}]}]}], ";", 
  RowBox[{"\[Beta]", " ", "=", " ", 
   RowBox[{"dbar", "/", "\[Omega]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Epsilon]", " ", "=", " ", 
   RowBox[{"RandomReal", "[", "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"cbar", " ", "=", " ", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"{", 
     RowBox[{"0", ",", "50"}], "}"}], "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"dbarmp", " ", "=", " ", 
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"d", " ", 
      RowBox[{"pdf", "[", 
       RowBox[{"n", ",", "d"}], "]"}]}], " ", ",", " ", 
     RowBox[{"{", 
      RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], "]"}]}],
   ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"dvarmp", " ", "=", " ", 
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"d", "-", "dbarmp"}], ")"}], "2"], " ", 
      RowBox[{"pdf", "[", 
       RowBox[{"n", ",", "d"}], "]"}]}], " ", ",", " ", 
     RowBox[{"{", 
      RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], "]"}]}],
   ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"cvdmp", " ", "=", " ", 
   RowBox[{
    SqrtBox["dvarmp"], "/", "dbarmp"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"cdf", "[", 
       RowBox[{"n", ",", "d"}], "]"}], ",", 
      RowBox[{"CDF", "[", 
       RowBox[{
        RowBox[{"GammaDistribution", "[", 
         RowBox[{
          RowBox[{"1", "/", 
           SuperscriptBox["cvdmp", "2"]}], ",", " ", 
          RowBox[{"dbarmp", " ", 
           SuperscriptBox["cvdmp", "2"]}]}], "]"}], ",", " ", "d"}], "]"}]}], 
     "}"}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"d", ",", " ", "0", ",", " ", 
     RowBox[{"5", " ", "dbarmp"}]}], "}"}], ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "}"}], ",", "Dashed"}], "}"}]}], ",", " ", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<d\>\"", ",", " ", "\"\<cdf(d)\>\""}], "}"}]}]}], 
  "]"}]}], "Input",
 CellChangeTimes->{{3.401534510137419*^9, 3.4015345659576845`*^9}, {
  3.401537385722307*^9, 3.401537390368989*^9}, {3.4017094311715136`*^9, 
  3.401709458971488*^9}, {3.401710066665309*^9, 3.4017100939245057`*^9}, {
  3.401710129065035*^9, 3.401710137967837*^9}, {3.4017101722070704`*^9, 
  3.401710276376859*^9}, {3.4017107508691454`*^9, 3.4017107510694337`*^9}, {
  3.411302179030208*^9, 3.4113021955639825`*^9}, {3.4304807298268175`*^9, 
  3.430480770655526*^9}, {3.4304808057159405`*^9, 3.430480825103819*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwV1nc4Vm8YB/AXrz3ekdKglOwGDVR03xqyMtqRUmZRdoXIDj+UsiKyVyoy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     "]]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], Dashing[{Small, Small}], 
    LineBox[CompressedData["
1:eJwV1Xc8lV8cB3Djcq3LvUYqiYyIhoYo43sICb8iNEiDS6JpZZdCyEiSSheZ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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox["\"d\"", TraditionalForm], 
    FormBox["\"cdf(d)\"", TraditionalForm]},
  AxesOrigin->{0, 0},
  PlotRange->{{0, 4.699808426020161}, {0., 0.9999092063247443}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 CellChangeTimes->{3.4304808258949566`*^9, 3.4304810693350067`*^9},
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["5.3 Out-of-Plane Pore Dimensions", "Section",
 Editable->False,
 CellChangeTimes->{{3.401790049996339*^9, 3.4017900561151376`*^9}, {
  3.412411345976043*^9, 3.4124113579532657`*^9}}],

Cell[TextData[{
 "Assuming all inter-fibre interfaces to define the boundaries of a void, and \
from the definitions of porosity, \[Epsilon],  and mean network thickness, ",
 Cell[BoxData[
  FormBox[
   OverscriptBox["z", "_"], TraditionalForm]]],
 StyleBox[", ",
  FontSlant->"Italic"],
 "we",
 StyleBox[" ",
  FontSlant->"Italic"],
 "obtain an expression for the mean pore height in terms of porosity, mean \
coverage, ",
 Cell[BoxData[
  FormBox[
   OverscriptBox["c", "_"], TraditionalForm]]],
 " and mean fibre thickness, ",
 Cell[BoxData[
  FormBox[
   OverscriptBox["t", "_"], TraditionalForm]]],
 ":"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.4124128587012353`*^9, 3.4124129999343185`*^9}, 
   3.412415819058019*^9}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"zbar", " ", "=", " ", 
   RowBox[{
    RowBox[{"hbar", " ", 
     RowBox[{"(", 
      RowBox[{"cbar", " ", "-", "1"}], ")"}]}], " ", "+", " ", 
    RowBox[{"cbar", " ", "tbar"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"htotbar", " ", "=", " ", 
   RowBox[{"hbar", " ", 
    RowBox[{"(", 
     RowBox[{"cbar", "-", "1"}], ")"}]}]}], ";"}], "\n", 
 RowBox[{"hbar", "=", " ", 
  RowBox[{"hbar", "/.", 
   RowBox[{
    RowBox[{"Solve", "[", 
     RowBox[{
      RowBox[{"\[Epsilon]", " ", "==", " ", 
       RowBox[{"(", 
        RowBox[{"htotbar", "/", "zbar"}], ")"}]}], ",", " ", "hbar"}], "]"}], 
    "[", 
    RowBox[{"[", "1", "]"}], "]"}]}]}]}], "Input",
 CellChangeTimes->{{3.4017849518555727`*^9, 3.4017850079962993`*^9}, {
  3.4017859363411937`*^9, 3.401785943952138*^9}, {3.401790002447968*^9, 
  3.401790033242248*^9}, {3.4017900968937745`*^9, 3.4017900977750416`*^9}}],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{"cbar", " ", "tbar", " ", "\[Epsilon]"}], 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "+", "cbar"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "+", "\[Epsilon]"}], ")"}]}]]}]], "Output",
 CellChangeTimes->{
  3.4017855712562275`*^9, {3.4017856014596577`*^9, 3.40178563672036*^9}, 
   3.401785989828104*^9, 3.401790033843112*^9, {3.401790078407192*^9, 
   3.4017901091914577`*^9}, 3.4017934748510385`*^9, 3.4017940588808317`*^9, 
   3.40220491562116*^9}]
}, Open  ]],

Cell[TextData[{
 "To take account of fibre contacts occurring at interfaces, we must include \
the fractional contact area, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", "\[Infinity]"], TraditionalForm]]],
 " as the probability of contact between vertically adjacent fibres. "
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.412413015877243*^9, 3.412413076554493*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"zbar", " ", "=", " ", 
   RowBox[{
    RowBox[{"hbar\[CapitalPhi]", " ", 
     RowBox[{"(", 
      RowBox[{"cbar", " ", "-", "1"}], ")"}], 
     RowBox[{"(", 
      RowBox[{"1", "-", "\[CapitalPhi]inf"}], ")"}]}], " ", "+", " ", 
    RowBox[{"cbar", " ", "tbar"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"htotbar", " ", "=", " ", 
   RowBox[{"hbar\[CapitalPhi]", " ", 
    RowBox[{"(", 
     RowBox[{"cbar", "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"1", "-", "\[CapitalPhi]inf"}], ")"}]}]}], ";"}], "\n", 
 RowBox[{"hbar\[CapitalPhi]", "=", " ", 
  RowBox[{"hbar\[CapitalPhi]", "/.", 
   RowBox[{
    RowBox[{"Solve", "[", 
     RowBox[{
      RowBox[{"\[Epsilon]", " ", "==", " ", 
       RowBox[{"(", 
        RowBox[{"htotbar", "/", "zbar"}], ")"}]}], ",", " ", 
      "hbar\[CapitalPhi]"}], "]"}], "[", 
    RowBox[{"[", "1", "]"}], "]"}]}]}]}], "Input",
 CellChangeTimes->{{3.4017849518555727`*^9, 3.4017850079962993`*^9}, {
  3.4017859363411937`*^9, 3.401785943952138*^9}, {3.401790002447968*^9, 
  3.401790033242248*^9}, {3.4017900968937745`*^9, 3.4017900977750416`*^9}, {
  3.401793385542619*^9, 3.4017934214542575`*^9}, {3.40179345612411*^9, 
  3.4017934600998273`*^9}, {3.4017940249420304`*^9, 3.4017940445702543`*^9}}],

Cell[BoxData[
 FractionBox[
  RowBox[{"cbar", " ", "tbar", " ", "\[Epsilon]"}], 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "cbar"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Epsilon]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[CapitalPhi]inf"}], ")"}]}]]], "Output",
 CellChangeTimes->{{3.4017934231867485`*^9, 3.4017934750112686`*^9}, 
   3.401794059071106*^9, 3.4022049158214483`*^9}]
}, Open  ]],

Cell[TextData[{
 "We derived an expression for ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", "inf"], TraditionalForm]]],
 " in Section 4.3.1, so we obtain the mean pore height in terms of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    OverscriptBox["c", "_"], " "}], TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox[
   OverscriptBox["t", "_"], TraditionalForm]]],
 " and \[Epsilon]:"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.412413082212629*^9, 3.4124131835683713`*^9}, 
   3.412413219500038*^9}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"\[CapitalPhi]inf", "=", " ", 
   RowBox[{"1", "+", 
    RowBox[{"\[Epsilon]", " ", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"2", "+", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", "3"}], "+", "\[Epsilon]"}], ")"}], " ", 
         "\[Epsilon]"}]}], ")"}], "/", 
      RowBox[{"Log", "[", "\[Epsilon]", "]"}]}]}]}]}], 
  ";"}], "\[IndentingNewLine]", "hbar\[CapitalPhi]"}], "Input",
 CellChangeTimes->{{3.412413206250987*^9, 3.412413216205301*^9}}],

Cell[BoxData[
 FractionBox[
  RowBox[{"cbar", " ", "tbar", " ", 
   RowBox[{"Log", "[", "\[Epsilon]", "]"}]}], 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "cbar"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Epsilon]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "3"}], "+", "\[Epsilon]"}], ")"}], " ", "\[Epsilon]"}]}],
     ")"}]}]]], "Output",
 CellChangeTimes->{3.4017938923513746`*^9, 3.4017940591211777`*^9, 
  3.4022049158815346`*^9}]
}, Open  ]],

Cell["\<\
As expected, we observe that the mean pore height is greater when fibre \
contacts are taken into consideration:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.412413245247061*^9, 3.412413282470586*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"cbar", " ", "=", " ", "10"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"LogPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"hbar", "/", "tbar"}], ",", 
     RowBox[{"hbar\[CapitalPhi]", "/", "tbar"}]}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"\[Epsilon]", ",", " ", "0.01", ",", " ", "1"}], "}"}], ",", " ", 
   
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"Dashed", ",", " ", 
      RowBox[{"{", "}"}]}], "}"}]}], ",", " ", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
     "\"\<\[Epsilon]\>\"", ",", " ", 
      "\"\<\!\(\*OverscriptBox[\(h\), \(_\)]\)\>\""}], "}"}]}]}], 
  "]"}], "\[IndentingNewLine]", 
 RowBox[{"cbar", "=."}]}], "Input",
 CellChangeTimes->{{3.4017941451448736`*^9, 3.401794199583152*^9}, {
   3.40179432555429*^9, 3.401794326065024*^9}, 3.401794372601941*^9, {
   3.401794586169035*^9, 3.4017945992077837`*^9}, {3.4017955877392225`*^9, 
   3.4017956267252817`*^9}, {3.4017956643293533`*^9, 3.401795684818816*^9}, {
   3.4017957634418707`*^9, 3.401795818661272*^9}, {3.4017958864687743`*^9, 
   3.4017959166121187`*^9}, {3.401796272614024*^9, 3.401796294715805*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], Dashing[{Small, Small}], LineBox[CompressedData["
1:eJwVUXk41HsfHWaGwWDGFooShRaVuOji802viixlS1fphiwlJNmTMJZsKZks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     "]]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], LineBox[CompressedData["
1:eJwV13k4VO0bB3DZsy/VK1mzU5RsFb6nhSJUkhalElJJlJRUSqLFkpJCil4p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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox["\"\[Epsilon]\"", TraditionalForm], 
    FormBox[
    "\"\\!\\(\\*OverscriptBox[\\(h\\), \\(_\\)]\\)\"", TraditionalForm]},
  AxesOrigin->{0.01, -4.4897572936624774`},
  FrameTicks->{{{{-4.605170185988091, 
       FormBox["0.01`", TraditionalForm]}, {-2.3025850929940455`, 
       FormBox["0.1`", TraditionalForm]}, {0, 
       FormBox["1", TraditionalForm]}, {2.302585092994046, 
       FormBox["10", TraditionalForm]}, {4.605170185988092, 
       FormBox["100", TraditionalForm]}, {-3.912023005428146, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-3.506557897319982, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-3.2188758248682006`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-2.995732273553991, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-2.8134107167600364`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-2.659260036932778, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-2.5257286443082556`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-2.4079456086518722`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-1.6094379124341003`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-1.203972804325936, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.916290731874155, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.6931471805599453, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.5108256237659905, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.3566749439387323, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.2231435513142097, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.10536051565782628`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {0.6931471805599453, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.0986122886681098`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.3862943611198906`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.6094379124341003`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.791759469228055, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.9459101490553132`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {2.0794415416798357`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {2.1972245773362196`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {2.995732273553991, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {3.4011973816621555`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {3.6888794541139363`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {3.912023005428146, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.0943445622221, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.248495242049359, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.382026634673881, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.499809670330265, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}}, {{-4.605170185988091, 
       FormBox["\" \"", TraditionalForm]}, {-2.3025850929940455`, 
       FormBox["\" \"", TraditionalForm]}, {0, 
       FormBox["\" \"", TraditionalForm]}, {2.302585092994046, 
       FormBox["\" \"", TraditionalForm]}, {4.605170185988092, 
       FormBox["\" \"", TraditionalForm]}, {-3.912023005428146, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-3.506557897319982, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-3.2188758248682006`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-2.995732273553991, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-2.8134107167600364`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-2.659260036932778, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-2.5257286443082556`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-2.4079456086518722`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-1.6094379124341003`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-1.203972804325936, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.916290731874155, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.6931471805599453, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.5108256237659905, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.3566749439387323, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.2231435513142097, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {-0.10536051565782628`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {0.6931471805599453, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.0986122886681098`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.3862943611198906`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.6094379124341003`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.791759469228055, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {1.9459101490553132`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {2.0794415416798357`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {2.1972245773362196`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {2.995732273553991, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {3.4011973816621555`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {3.6888794541139363`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {3.912023005428146, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.0943445622221, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.248495242049359, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.382026634673881, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {4.499809670330265, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}}}, {Automatic, Automatic}},
  GridLines->{None, None},
  Method->{},
  PlotRange->{{0.01, 1}, {-4.4897572936624774`, 4.9213748266550095`}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]},
  Ticks->{Automatic, {{-4.605170185988091, 
      FormBox["0.01`", TraditionalForm]}, {-2.3025850929940455`, 
      FormBox["0.1`", TraditionalForm]}, {0, 
      FormBox["1", TraditionalForm]}, {2.302585092994046, 
      FormBox["10", TraditionalForm]}, {4.605170185988092, 
      FormBox["100", TraditionalForm]}, {-3.912023005428146, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-3.506557897319982, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-3.2188758248682006`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-2.995732273553991, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-2.8134107167600364`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-2.659260036932778, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-2.5257286443082556`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-2.4079456086518722`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-1.6094379124341003`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-1.203972804325936, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-0.916290731874155, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-0.6931471805599453, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-0.5108256237659905, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-0.3566749439387323, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-0.2231435513142097, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {-0.10536051565782628`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {0.6931471805599453, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {1.0986122886681098`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {1.3862943611198906`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {1.6094379124341003`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {1.791759469228055, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {1.9459101490553132`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {2.0794415416798357`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {2.1972245773362196`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {2.995732273553991, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {3.4011973816621555`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {3.6888794541139363`, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {3.912023005428146, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {4.0943445622221, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {4.248495242049359, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {4.382026634673881, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}, {4.499809670330265, 
      FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
       Thickness[0.001]}}}}]], "Output",
 CellChangeTimes->{{3.4017941734555826`*^9, 3.4017942004443903`*^9}, 
   3.4017943270163918`*^9, 3.401794373343006*^9, {3.401794587140432*^9, 
   3.401794599438115*^9}, {3.401795589601901*^9, 3.401795596231434*^9}, 
   3.401795627486376*^9, {3.4017956648000307`*^9, 3.4017956855999393`*^9}, {
   3.4017957639626193`*^9, 3.401795818881589*^9}, {3.401795887289955*^9, 
   3.401795916872493*^9}, {3.4017962742463713`*^9, 3.401796318630192*^9}, 
   3.402204919576848*^9}]
}, Open  ]],

Cell[TextData[{
 "The increase in ",
 Cell[BoxData[
  FormBox[
   SubscriptBox[
    OverscriptBox["h", "_"], "\[Infinity]"], TraditionalForm]]],
 "observed as \[Epsilon] \[Rule]0 arises as an artefact of the model. If we \
assume the theory is applicable in the range that ",
 Cell[BoxData[
  FormBox[
   SubscriptBox[
    OverscriptBox["h", "_"], "\[Infinity]"], TraditionalForm]]],
 "increases with porosity, then we identify the lower limit of application of \
the theory as:"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.412413325141944*^9, 3.412413428079962*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"dhbar\[CapitalPhi]", "=", " ", 
   RowBox[{"Simplify", "[", 
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"hbar\[CapitalPhi]", "/", "tbar"}], ",", " ", "\[Epsilon]"}], 
     "]"}], "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{"dhbar\[CapitalPhi]", " ", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"cbar", "-", "1"}], ")"}], "/", "cbar"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"\[Epsilon]", ",", ".2"}], "}"}]}], 
  "]"}], "\[IndentingNewLine]", 
 RowBox[{"\[CapitalPhi]inf", "/.", "%"}]}], "Input",
 CellChangeTimes->{{3.4017959266165037`*^9, 3.401795997298139*^9}, {
  3.4017961334338927`*^9, 3.401796135526902*^9}, {3.4017970563309536`*^9, 
  3.401797056561285*^9}, {3.401798055728016*^9, 3.4017980656022143`*^9}}],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\[Epsilon]", "\[Rule]", "0.20558918557665212`"}], "}"}]], "Output",\

 CellChangeTimes->{3.4017980756967297`*^9, 3.402204920398029*^9}],

Cell[BoxData["0.8147342946706054`"], "Output",
 CellChangeTimes->{3.4017980756967297`*^9, 3.402204920428072*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["5.4 Porous Anisotropy", "Section",
 Editable->False,
 CellChangeTimes->{{3.4021310488159103`*^9, 3.402131052561296*^9}, {
  3.4124134741462016`*^9, 3.4124134750474977`*^9}}],

Cell["\<\
A plot of our expression for mean pore diameter in units of the fibre width \
against the mean pore height in units of fibre thickness reveals an \
approximately linear relationship:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.4124135419136467`*^9, 3.4124135933776484`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"hbar", " ", "=", " ", 
   RowBox[{"cbar", " ", "tbar", " ", 
    RowBox[{
     RowBox[{"Log", "[", 
      RowBox[{"1", "/", "\[Epsilon]"}], "]"}], "/", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"cbar", "-", "1"}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{"1", "-", "\[Epsilon]"}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{"2", "-", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"3", "-", "\[Epsilon]"}], ")"}], " ", "\[Epsilon]"}]}], 
        ")"}]}], ")"}]}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"dbar", " ", "=", "  ", 
   RowBox[{"2", " ", 
    RowBox[{"\[Omega]bar", "/", 
     RowBox[{"Log", "[", 
      RowBox[{"1", "/", "\[Epsilon]"}], "]"}]}]}]}], ";"}], "\n", 
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"hbar", "/", "tbar"}], ",", " ", 
      RowBox[{"dbar", "/", "\[Omega]bar"}]}], "}"}], "/.", 
    RowBox[{"cbar", "\[Rule]", 
     SuperscriptBox["10", "6"]}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"\[Epsilon]", ",", " ", ".21", ",", " ", ".9"}], "}"}], ",", " ", 
   
   RowBox[{"AspectRatio", "\[Rule]", 
    RowBox[{"1", "/", "GoldenRatio"}]}], ",", " ", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
     "\"\<\!\(\*OverscriptBox[\(h\), \(_\)]\)/\!\(\*OverscriptBox[\(t\), \
\(_\)]\)\>\"", ",", 
      "\"\<\!\(\*OverscriptBox[\(d\), \(_\)]\)/\!\(\*OverscriptBox[\(\[Omega]\
\), \(_\)]\)\>\""}], "}"}]}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], "Input",
 CellChangeTimes->{{3.4021320625335646`*^9, 3.402132085797016*^9}, {
  3.4021321425686493`*^9, 3.402132142859067*^9}, {3.4021331756341248`*^9, 
  3.402133225475794*^9}, {3.4021335770313053`*^9, 3.4021335830800033`*^9}, {
  3.4021343760302095`*^9, 3.402134381317813*^9}, {3.430481347184534*^9, 
  3.4304813478254557`*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwV0ns81HkXB3D3IdW4FYWSCdkaPBO1mm3PSTdPFNlEWJNEhaJ1K7dMqcit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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox[
    "\"\\!\\(\\*OverscriptBox[\\(h\\), \
\\(_\\)]\\)/\\!\\(\\*OverscriptBox[\\(t\\), \\(_\\)]\\)\"", TraditionalForm], 
    
    FormBox[
    "\"\\!\\(\\*OverscriptBox[\\(d\\), \\(_\\)]\\)/\\!\\(\\*OverscriptBox[\\(\
\[Omega]\\), \\(_\\)]\\)\"", TraditionalForm]},
  AxesOrigin->{2., 5.},
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.4021335956981473`*^9, 3.402134391692731*^9, 
  3.4022049313738112`*^9, 3.4022053502861776`*^9, 3.4022093376997967`*^9, 
  3.4022093814026384`*^9, 3.402211280954059*^9, 3.4022176576332736`*^9, 
  3.402217710639493*^9, 3.4022179528277426`*^9, 3.4304813559170914`*^9},
 ImageCache->GraphicsData["CompressedBitmap", "\<\
eJzNmglwFEUUhifZDRDOBEIEQYgHRaRUxAODB0SQKwSQokBKRAgIlagIcpU3
URE8QFS8DyLKDQpeIAoEQS5FoyAGFY2RQigLTGnUEo9a83dPb3fvvLe9aizZ
qkxmp+d73e/o169np3/BlMKx4wumFI0pyOo9qWBiYdGYyVm9JkyquRRK8ryk
Qs/zDmR5OI/UnPqHtl4kEvFC8kvNmfjDsbc49WqlXTTXptQ6sj2tuvpYccn8
XeJbZmJMvWPH/qjq17cE/51MXdn+xJzZWwpfWb1XfGueGJN16NCP5SOGLxP9
uJh6sh19oC/xLcPBpMr2klkzN41au2af+NYsMQZjWjt1yprcsg8POpn62tal
E2547ewvPj/iNXUwDTSzvahwdZ+dO77x0slIaKhHtOrWW9YJTdIc0htpBtrP
nffINq+Jg/HbJ65Yvhv9YGRKDsv447h8y+aKsjGjVwrNGzqYdO19aD5sw/r9
yhosI60Zhj5L7ixejwjAeaoDy9BBg+FlV1ZWqUBiGT8QcS8YmEMFLMtk6uHB
1AgcmC/FgbWQ7fA95oGIgbCDaSm7anH06M/oBt5FV8lk6LQS/1IQMuggZ+8n
h3HFT241J33lfa21ULhk1lNP7ozE3NJGjuumRQvL0CRF4Vof2S7zURhtSgTM
IW7pZd6SolSDKFhYi+pp3lfXNAG8YYqV9/cw77dNlqyHW3HFkMXCtKKlO8+E
gqbFuWjM5THfy8rIMtPgWjeekZk2DM+hG3QX7eoSAxNUPTtMoJOX64/YFNoP
x2Rc6R89u4ofQ6oeA8wK88LMXQnJefJW61rNpx99eTjfp5znYbgcqih3XojL
4Tgi/35PjYgA6BKjGj75tOT8RCzY2A4YZE2c50RZISUUp5s4ChipFaYSeegC
28OU//9WH+k6BJC3VAh0JoTn20q5TNMs6Gj0IhrbE+Lr8KKaa09CFAzisXJS
eDmZ2lvwFDyGNUg0tiFECRWTuYYQ35FM62E1X1F+RVNha9pHIfqyXAw88WWw
TgWJ4v4YqcU1nY8465rfIQxlhji1slEiE+0mUfb/6iNReWnRuaGM7zcm/2ce
ksVnGNMCiQ7TBNEWOo6sd9x5yHCE//HNZV02EkDCHpKJRFSCyHmob7CFoHZR
FJ5Ty/flx6gldQ+qGiebZUkI6wI2Nyi/kQGy+d6c13ShUocYX5yxnCKvYcsQ
rYREy0k4hAx/UfSpUhNV8EACfCMa1aakoqIit7S0FH9eSB6zy8vLozLirFPt
aEMZ6wuJyVUsDJ3M+WusFiSWHQw0LDiisRWPddCDNPO5aGxpYII6Q96sUou1
kJ3A93GmHhoQtbURjZk81lFiaqnBH4YpGjN4rJPE0AOMYG08mvLYOXqQqkaE
C4xgILHzJAabwcFwdHSQTXhMVlRhhBvMgWIrWg414jF/mqP2s7cWDXmmix4h
thao6bAtF431eewiiaFggxmxw44GRj0ek5sVscOOVqmipW5sMHXVsw/qW7Mv
TtWWa88qK2DjZIrucUI3icd6BEMXIiI80VNHLcamAuJPnuhNx8LvPNFXhwGU
Ec/ZfuNvz9OWhhvVGvQrT/i7E8SJmbp+4YkBdtZS9v2JJwbapgUF01bHxskg
Hb/KoBjXD7zkwdqgKnShexVPDNFRi3GIqP2ev31oMHSRbo7wxDAdtSrNIkK+
44krdcCaG6bDPDHczmCgoNAhnhihTWsGxkGeGGkHhtLjAE8U0HpU8sRoWo+v
YwPjGnr0X/GSx9Gj388ThfToP+eJInr0n/HEdUE9xDaZJ8ZrPZDO1fTcyxMT
6Om5hycmBqcn6N08MUnrgRFhZNDjI0mIpVdXZGa1Z0mZIqWANHX7UErxHyNY
9aFH78qpDc802gy7/plw8XBzGm2n92tlvDfTJt1ZK8JvtS2tHm9urxXht0tc
PQRVVexWIxj0YxM2GO6QUswKHVPxXT4IiyVhLr+YJpt54i5JmKsFzLCJJ2ZI
QtU4SreNPHEPrccGnphF6/E2T9xH67HOIATwAD36tbzk2fTo1/DEg/ToX+eJ
h+jRv8oTj9B6rOaJebQeq3jisaAemC4v8cQTWg8zfa3giafonLSMJ56mU85S
nniWziOLeWK+nRyUHgtjo+l5evQv8JIX0KNfwBMv0qMv4YlF9Oif44kltB7P
8MRSWo+neWJ5UA/EyZM8sVLrYRYsj/PEy5KILVge5YlVWg+zYJnHE69oPczC
6+HY2HhN3hdbbs3lJb9hj15thebwxBo9erUdwv/ZPPGmzgJm8X4/T7wlCVW8
qwcR9/LEekmoJ/4qm83kiY10NpvBE6V0NrubJ94JZjP45k6e2KKzGbys5tF0
nthKz6PbY2NjGz17buMl76Bnzy088R49e27iiV307JnGEx/Qs2cqT5QFZw/i
ZDJPfKznkRl/N/LEHjv+EOfoT+4CjFJc/aCkP1QVJvcVYfXUScWm3IWol2Go
itBZJX5Kx/D1/17yvmCsIw6u/QeSRZn/mZ4J5opSaMqzi+VInLp2Pz1LxpnS
UmKlie/UmydfBucSYnCs4e3kqLPFGfVzXUUwLqHtGD7KKnVcqvmFuCzgiQP2
/FL5fSRPyGcRITO3e1fzt3+rAwrC1UOlOD9MHw4GChS6kie+06FglqrDeEI+
GAqrx0vK3UMNQgBHg45EiAzhJVdppyEkldMG88QPQadB90E8UW07TSXFgTzx
k9ZDPUuH4wbwxC+239SinM8T8umltShDpzyekE9Ioz8MKL/14Yk/tN/MxNeL
J/4MJjR4sCdPRHT8mcWt+SqTAPyEIm6OrWsvdYgP2+lGebCbA0ux3ajWtq4O
rC6dSC52YKl2YKpV7kIH1kB71UwpOQ5M/sgSNvMK9OzswBoH4xTz4nwHlkYn
mXMdWFM70yh3d3JgGba7VbrpSIZU82DOwUObsxx9nKCdjAdHyslnOLCWdPbp
4MBa2SlIOTnbgbXWupmObu/A2tDJqJ0Dy7IzknLyqQ7sFNvJKi2d7MBOo3NT
WwfWzk5QaoGM83u0OLbX7obPkHwgwvw9WofU6VKKeLtWSYnzE7Q41ua78/9h
uzgv+vfn0nB2sSmO/vp1Etfuv30rZq2oEKlXvSL+y5Pi7DKD4N4ocz7mjUrr
YUojXoOjXiZxCcuMNzQbi70WtYu/BIq0FIqjGFXlmwJOJBVjGPMV7FjtR5WU
lBRPnz7dSxFH8SaJTXcLhoX/WpBiJcnxeUHeS/oLzS/MAw==\
\>"]]
}, Open  ]],

Cell[TextData[{
 "The maximum value of ",
 Cell[BoxData[
  FormBox[
   OverscriptBox["h", "_"], TraditionalForm]]],
 "/",
 Cell[BoxData[
  FormBox[
   OverscriptBox["d", "_"], TraditionalForm]]],
 " is"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.4124136302907267`*^9, 3.4124136517215424`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Limit", "[", 
  RowBox[{
   RowBox[{"hbar", "/", "dbar"}], ",", " ", 
   RowBox[{"\[Epsilon]", "\[Rule]", "1"}]}], "]"}]], "Input"],

Cell[BoxData[
 FractionBox[
  RowBox[{"cbar", " ", "tbar"}], 
  RowBox[{"2", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "cbar"}], ")"}], " ", 
   "\[Omega]bar"}]]], "Output",
 CellChangeTimes->{3.40213427930112*^9, 3.4021343921333647`*^9, 
  3.4022049325054383`*^9, 3.4022053506667247`*^9, 3.402209338060315*^9, 
  3.402209381863301*^9, 3.4022112818954124`*^9, 3.4022176633514957`*^9, 
  3.4022177110400686`*^9, 3.402217953198275*^9, 3.430481371229109*^9}]
}, Open  ]],

Cell[TextData[{
 "Assuming in-plane pore diameters to be distributed according to a gamma \
distribution, we define the probability density for the smallest of ",
 StyleBox["n",
  FontSlant->"Italic"],
 " in-plane pore diameters (",
 StyleBox["fdn[n]", "Input"],
 ") and the corresponding cumulative distribution function (",
 StyleBox["Fdn[n]", "Input"],
 ") using the expressions obtained in Section 5.2:"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.412413748040042*^9, 3.412413867692093*^9}, {
  3.4124139220802994`*^9, 3.412413924503784*^9}}],

Cell[BoxData[{
 RowBox[{"Clear", "[", "hbar", "]"}], "\n", 
 RowBox[{
  RowBox[{"fd", " ", "=", 
   RowBox[{"PowerExpand", "[", 
    RowBox[{"PDF", "[", 
     RowBox[{
      RowBox[{"GammaDistribution", "[", 
       RowBox[{"\[Alpha]", ",", 
        RowBox[{"\[Beta]", "/", "\[Alpha]"}]}], "]"}], ",", "d"}], "]"}], 
    "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Fd", " ", "=", " ", 
   RowBox[{"FunctionExpand", "[", 
    RowBox[{"CDF", "[", 
     RowBox[{
      RowBox[{"GammaDistribution", "[", 
       RowBox[{"\[Alpha]", ",", 
        RowBox[{"\[Beta]", "/", "\[Alpha]"}]}], "]"}], ",", "d"}], "]"}], 
    "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"fdn", "[", "n_", "]"}], " ", ":=", " ", 
  RowBox[{"n", " ", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", "Fd"}], ")"}], 
    RowBox[{"n", "-", "1"}]], "fd"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Fdn", "[", "n_", "]"}], " ", ":=", " ", 
  RowBox[{"1", "-", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", "Fd"}], ")"}], "n"]}]}]}], "Input",
 CellChangeTimes->{3.402205320853856*^9}],

Cell[TextData[{
 "Similarly, for the pore heights we assume an exponential distribution and \
we have the probability density for the smallest of ",
 StyleBox["n",
  FontSlant->"Italic"],
 " in-plane pore heights (",
 StyleBox["gdn[n]", "Input"],
 ") and the corresponding cumulative distribution function (",
 StyleBox["Gdn[n]", "Input"],
 "):"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.41241389235756*^9, 3.4124139476270337`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"gd", " ", "=", 
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"ExponentialDistribution", "[", 
      RowBox[{"1", "/", "hbar"}], "]"}], ",", "d"}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Gd", " ", "=", " ", 
   RowBox[{"Refine", "[", 
    RowBox[{
     RowBox[{"CDF", "[", 
      RowBox[{
       RowBox[{"ExponentialDistribution", "[", 
        RowBox[{"1", "/", "hbar"}], "]"}], ",", "d"}], "]"}], ",", " ", 
     RowBox[{"d", ">", "0"}]}], "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"gdn", "[", "n_", "]"}], " ", ":=", " ", 
  RowBox[{"n", " ", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", "Gd"}], ")"}], 
    RowBox[{"n", "-", "1"}]], "gd"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Gdn", "[", "n_", "]"}], " ", ":=", " ", 
  RowBox[{"1", "-", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", "Gd"}], ")"}], "n"]}]}]}], "Input",
 CellChangeTimes->{{3.4021378884408083`*^9, 3.402137924122115*^9}, {
  3.402137960444344*^9, 3.4021380132803183`*^9}, {3.402138088959139*^9, 
  3.402138143597706*^9}, {3.40213822706773*^9, 3.4021382284697456`*^9}, {
  3.4021383881193104`*^9, 3.402138399856187*^9}, {3.4021385149416723`*^9, 
  3.4021385525757875`*^9}, {3.402138660671221*^9, 3.4021386618328915`*^9}, {
  3.4021387108834224`*^9, 3.4021387272068944`*^9}, {3.402138786131624*^9, 
  3.402138812008834*^9}, {3.4022048332126627`*^9, 3.4022048518895187`*^9}}],

Cell["\<\
The probability density of the smallest pore dimension in a path consisting \
of alternate pore heights and pore diameters is therefore:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.412413956680051*^9, 3.412413998910776*^9}, 
   3.4124158256474943`*^9}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"pd", " ", "=", " ", 
  RowBox[{"Simplify", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", " ", "-", " ", 
       RowBox[{"Gdn", "[", 
        RowBox[{"n", "-", "1"}], "]"}]}], ")"}], 
     RowBox[{"fdn", "[", "n", "]"}]}], " ", "+", " ", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", "-", 
       RowBox[{"Fdn", "[", "n", "]"}]}], ")"}], 
     RowBox[{"gdn", "[", 
      RowBox[{"n", "-", "1"}], "]"}]}]}], "]"}]}]], "Input"],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"d", "/", "hbar"}]], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"-", 
      FractionBox["d", "hbar"]}]], ")"}], "n"], " ", 
  RowBox[{"(", 
   RowBox[{
    FractionBox[
     RowBox[{
      RowBox[{"-", "1"}], "+", "n"}], "hbar"], "+", 
    FractionBox[
     RowBox[{
      SuperscriptBox["d", 
       RowBox[{
        RowBox[{"-", "1"}], "+", "\[Alpha]"}]], " ", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"-", 
        FractionBox[
         RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}]], " ", "n", " ", 
      SuperscriptBox["\[Alpha]", "\[Alpha]"], " ", 
      SuperscriptBox["\[Beta]", 
       RowBox[{"-", "\[Alpha]"}]]}], 
     RowBox[{"Gamma", "[", 
      RowBox[{"\[Alpha]", ",", 
       FractionBox[
        RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}], "]"}]]}], ")"}], " ", 
  SuperscriptBox[
   RowBox[{"(", 
    FractionBox[
     RowBox[{"Gamma", "[", 
      RowBox[{"\[Alpha]", ",", 
       FractionBox[
        RowBox[{"d", " ", "\[Alpha]"}], "\[Beta]"]}], "]"}], 
     RowBox[{"Gamma", "[", "\[Alpha]", "]"}]], ")"}], "n"]}]], "Output",
 CellChangeTimes->{3.402209382343992*^9, 3.4022112866923103`*^9, 
  3.402217664403008*^9, 3.40221771152076*^9, 3.4022179537190237`*^9, 
  3.430481737175314*^9}]
}, Open  ]],

Cell["\<\
We input  the relationships obtained earlier for the mean and variance of \
pore diameter and pore height:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.4124140325491457`*^9, 3.4124140822506127`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{"dbar", " ", "=", " ", 
   RowBox[{"2", " ", 
    RowBox[{"\[Omega]bar", "/", 
     RowBox[{"Log", "[", 
      RowBox[{"1", "/", "\[Epsilon]"}], "]"}]}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"hbar", " ", "=", " ", 
   RowBox[{"cbar", " ", "tbar", " ", 
    RowBox[{
     RowBox[{"Log", "[", 
      RowBox[{"1", "/", "\[Epsilon]"}], "]"}], "/", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"cbar", "-", "1"}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{"1", "-", "\[Epsilon]"}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{"2", "-", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"3", "-", "\[Epsilon]"}], ")"}], " ", "\[Epsilon]"}]}], 
        ")"}]}], ")"}]}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Beta]", " ", "=", " ", "dbar"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Alpha]", " ", "=", " ", 
   RowBox[{"1", "/", 
    SuperscriptBox["cvd", "2"]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"cvd", " ", "=", " ", 
   RowBox[{
    SqrtBox[
     RowBox[{"16", "-", 
      SuperscriptBox["\[Pi]", "2"]}]], "/", "\[Pi]"}]}], ";"}]}], "Input",
 CellChangeTimes->{{3.40213908855649*^9, 3.4021390891773825`*^9}, {
   3.4021391407916*^9, 3.4021391478217087`*^9}, 3.4021396545203056`*^9, {
   3.402139881556768*^9, 3.402139965377296*^9}, {3.402140000678056*^9, 
   3.4021400578502655`*^9}, {3.4021401423417587`*^9, 3.4021401497424*^9}, {
   3.4022056786082816`*^9, 3.4022056982064624`*^9}, 3.402205849934637*^9, {
   3.4304817407504544`*^9, 3.4304817413813615`*^9}}],

Cell["\<\
We now compare the distributions of pore height (dashed lines) and in-plane \
pore diameter (solid lines) for networks formed from fibres with circular \
cross section with porosity \[Epsilon] = 0.8 (first plot) and \[Epsilon] = \
0.5 (second plot). It is clear that the smallest pore dimensions in a path \
from one side of the network to the other are likely to be associated with \
the height of pores rather than their in-plane dimensions.\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.4124140790560193`*^9, 3.4124141832358227`*^9}, {
  3.4124142933842077`*^9, 3.4124143909344783`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"tbar", " ", "=", " ", 
   RowBox[{"\[Omega]bar", " ", "=", " ", "1"}]}], ";", 
  RowBox[{"\[Epsilon]", " ", "=", " ", "0.8"}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"fd", ",", 
     RowBox[{"Limit", "[", 
      RowBox[{"gd", ",", " ", 
       RowBox[{"cbar", "\[Rule]", "\[Infinity]"}]}], "]"}]}], "}"}], ",", " ",
    
   RowBox[{"{", 
    RowBox[{"d", ",", " ", "0", ",", " ", "30"}], "}"}], ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "}"}], ",", "Dashed"}], "}"}]}], ",", " ", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
     "\"\<\!\(\*OverscriptBox[\(d\), \
\(_\)]\)/\!\(\*OverscriptBox[\(\[Omega]\), \(_\)]\)\>\"", ",", 
      "\"\<f(d) or g(d)\>\""}], "}"}]}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Epsilon]", "=", " ", "0.5"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"fd", ",", 
     RowBox[{"Limit", "[", 
      RowBox[{"gd", ",", " ", 
       RowBox[{"cbar", "\[Rule]", "\[Infinity]"}]}], "]"}]}], "}"}], ",", " ",
    
   RowBox[{"{", 
    RowBox[{"d", ",", " ", "0", ",", " ", "10"}], "}"}], ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "}"}], ",", "Dashed"}], "}"}]}], ",", " ", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
     "\"\<\!\(\*OverscriptBox[\(d\), \
\(_\)]\)/\!\(\*OverscriptBox[\(\[Omega]\), \(_\)]\)\>\"", ",", 
      "\"\<f(d) or g(d)\>\""}], "}"}]}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}], "\[IndentingNewLine]", 
 RowBox[{"Clear", "[", 
  RowBox[{"tbar", ",", " ", "\[Omega]bar", ",", " ", "\[Epsilon]"}], 
  "]"}]}], "Input",
 CellChangeTimes->{{3.4022057384743648`*^9, 3.4022058181088734`*^9}, {
  3.402206303937461*^9, 3.4022063822901263`*^9}, {3.4022064566971183`*^9, 
  3.402206633661581*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwV1Hc8198XB3B87JVNZsPIiEi2zrFSyqiotEgUFZFSicwiI4WsluxI4fMm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     "]]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], Dashing[{Small, Small}], 
    LineBox[CompressedData["
1:eJwVz38803kcB3CXCvdoRVlspCubavk1P74zbO8PolZnIiz9kEQdoWtKqXuU
X9Xy44phcp1jcSjFJEWKpl9IxVWTRFRu6rAdq4Tc5/54P16P5+Px+uP1Xha6
zy98lpaWFh/f//mpt+UuLa2Yu3taqCgXtcGzAuN4t64kSM3x0Ko/0gaI/CTU
uCsLekrnsDqi2iBu40WFdpcU2s8YNRn6toHBiFHWhKIaDsiVPaOUNuhr4O8b
V8hh6YLnjLGKVvCifXitVnRCxeyYO+SuFphQsHJUigHo5PZ3kNwfQrNHaXGH
dAzaI32LXOj3wLSvbjDUchrEDLsVrefkYBi146KOhzYqXTBVuX5PE2QxvW7y
A3SRW3NM6NQfDfAyz7ZJrU9CwyzdHh+HOkgrPv5kFsUAmfAigg0018Ao++1p
8e+LUKCHTjrX6iqc8udxOz+RUVoIkOWDVRDLLtrRkmSMblsOpariL8Oa0a9H
Yt5S0VD1+wEvVTkkuvrupDotQYYpNXN1Bv4E5yvnao41LkUfOp/0rpy4AHu/
qBMekpahC6Q7d9olRfBltX6vVcByFHwry3HexQJwuVRRmJtrjqK82N7xrfmg
Wrg+JFBOQ5tNpq4MF0jgbXpIoFDHAs0VTqco87Nh3egh3ntiBaq1GChJMc+C
j7t0Y2uOr0Qmp09ua847A540o1mn5KtQx5inQJGbBgbtFr9WjTJQESW2PdJH
BCaiyo0EYYk+mvkHFjNPgGQ1W3/RHivkmVb0YkdJErRQ3RzHMq3R0c1KKkWY
AGxT+VmbFzYo8OVBcd/VX2C+tvTAziRbJBWfUNR7xYNZ4lLXIAsmSpZksP6e
Ewd69G21fg+YSCNi+l5wjQVrDqXy0HY7JBHt1R0d/xmEvQub12vZI+7hl6vr
xNEwOpYewMuzR2WU1xGDOpGw1uZ+Ps/KAcmeRlNkn3fD7TGXyc+PHRDfsEfZ
Mi8MYiztujvCHVEmnWVTX74TOCNhz4L0CLRueUZ2WGsw/t/YmyQlkNrbjeJA
3gorE1IPy1ksVLtkA72iXwCMcHYjvZuFZuZ7Dr157g/ht39KqdnvhIIXmaYv
3uwHjYXPwtRkNipLNu09qvGB8bKZzzbX2WjrDzJpu3ID7KJf0sv3dka61/ju
J/LWgZohJqpVzmhLXU4rTegJN4on+zUiFxREbUt81OsOW9aYIxczVySiC1SC
EgQm0afqrJtc0fjR8IYt8zlA/j5K5i3goPvibd82SdhgbZtxlzrJQa3mdjnD
PxJAe+8bZI0tkxWUnsGm5i8eccN+8Dr3JhN7zuxCowjs3Z9WKQ9uIKBHUR15
HdsitnvTNx4Bqce79P2nOChgKPjggrUEDD42354xzUEx7g/f2LnhfrJSXYit
N/kb7TkioNPp8ska7KrErxGHsG9JiapX2AF9gombQEB2HE+b8Y2DrtHFDE8u
Ae5m+8rvY5+1nasQuBDg9Jc9txub1phs/9WZAGvRl85hbPXVtMzz2JR/E6cM
ZziIUevs188mQHUv22cXdsIew8FIJ7zvSNC7OGzN/nd8EnaPjVl8KnbzFW5d
JYuAB+dKpTJsv5LMrHGCgAZ+FOsetrFf6ncS7Gpt5qMubI4RScjGLruhCfkH
22mE8u6VIwEF0fWaGezzT6sEx7D/AydoPG8=
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox[
    "\"\\!\\(\\*OverscriptBox[\\(d\\), \\(_\\)]\\)/\\!\\(\\*OverscriptBox[\\(\
\[Omega]\\), \\(_\\)]\\)\"", TraditionalForm], 
    FormBox["\"f(d) or g(d)\"", TraditionalForm]},
  AxesOrigin->{0, 0},
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.402209385969205*^9, 3.402211289085752*^9, 
  3.4022176666462336`*^9, 3.402217715296189*^9, 3.4022179572040353`*^9, 
  3.4304818108011827`*^9},
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwV1Xk0VesbB3BTiLhIbmQeUgopSknfLSSRRpGkTBlKpRuqkxLdZCiSjCFC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     "]]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], Dashing[{Small, Small}], 
    LineBox[CompressedData["
1:eJwVzns803scx/Etl3LJsYpyyTaTslDOI10enXy+0anEUR3kHB7HZT9RhlpN
bifHWes4E+ZRjkxpdih1Mh25nDPJYSRqxNYp2mKj6ZySo5ylQs73/PF+vB/P
/1501pEvDy4gkUhBeP9/WRbL1Cv2jE/Su162Ya8OOlS0wDDaYbhp//jQxd06
mO9sOUWjpcPn7735O7froKw1n2pCE0DAGql15ac6SNyWkjRNFQE7+qcIvo0O
VDOHiiap12AwVn5uQK2FheqR1a+oMhj6uCGrM0ELkntizktqNzQ6r7EadR6G
mhsCdaLnEzgomDtu26qGeOgXjR8bBWF0QY/Z+AAk691G62v+hp6Si2Rd0iMw
p/NXD2+eBO41C3sR+SEUWN9nSqKnQG19kLt3ph9Y6pA/jBRvoXTzhWapcR9w
7vBZR1Z/gD1tb4W7RAqoPWoZvMJ/Djo9fKprZd0gyWS/WB9GQkuUGRWh/3QC
Tzy7/99xMiq4Ie/MK+0AkkqLmjOMUL3fncCkPDm4cuPSQpgmaDq4/kNHXis8
M5w/YHHbFE1lmQwS8tuwR59w8ixnEVp47qlewm+CoGeFa52czJHm9FaWcc1v
8KKdld0ltUDD1WEsSVADeFIerfSLWow2JTbmcUl1YA6GGM2UFZoqPCCLWFoL
u16FTzuHWqOX/Nfe5Qk1MCCPJYe7U1DRTa7drVXXoeHHjkiagYLu8spO+Sdc
hTiTFG1v7xJ0tD1qGWq/DMSxzAbL0qVo5lWplXFeBayLX9sbGLMM8StseCYl
EniwxN0iZYsNksiUE0OEGG7V8C64G9miVZXCAv6mi5BJZu5TPLVFg+ylvmcY
IjDXZOxOrl+O9qezJso5xaC/Qi8Sf7sCObbev2FYXwRFa+yMZaF2yO+Nq3M/
6SwUlsbdjaHZox15NsIB90IIL/qm1/K5PaLsVPJP0vLh3Y7L78UtDoitWWDc
NyeAUb1vmjzHEflvCGEbReXAz9u8HHpiVqKJVINmXsUH6ekUSryHE7Kry27c
WM+DnFAiwMjghPqL+Zy3D7PBWyO5ltNDRd9PFo3EmmZBidsFaU8ADXWU0JKs
ZjPgu9b8kgAZDbVxui2CzdLgC4fQy/dc6Khm1VrJJW4KbE++G0ER0NF3zCdX
4wOPg9iR5aIw0BGP5AJeGzjg0l5VHBThjOpavO2y/kqGQg85U9nijGbnuuJk
SjbETRn3dzMZaG/qL9wO5SFg1b6hCYQMZPZJTeXC/XHQdW/RqeZZBvJxUykS
RwhI9NMVdmMLLMnFrtjWzbJLj7B7hlyjtToCvpImNE1i5/iGTwdjPxcqXrvM
MZBtVZTXVi0BpqGF0fnYvWM6ldkQAX7Dy30iPzKQPqRaWTVIwFjYZAAbuyG1
vykGO7ev6+s0bIbjRIUDtrItPeUstodmcbpwgABWpbq6E/v04Jhn6mMCsg+L
HdbNM1CXYqZr558EuIykun2GPX7kfCsJuyt83yZ/bHokU9b0EPcHkoMJbO3i
jVJPbLEnkXseWxTxrNxWRYBv1daSSuzk6S3lfUrcR112pRb7g/yH8lxsT8qd
tvvYV9UmlR/7cZ+g7MEAtoG+uep37BMLTjzVY0fmxlw/hm2fGfTyDTaXwvvV
HbtlyvX9PLayTtQ41kfAf6i/Us4=
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox[
    "\"\\!\\(\\*OverscriptBox[\\(d\\), \\(_\\)]\\)/\\!\\(\\*OverscriptBox[\\(\
\[Omega]\\), \\(_\\)]\\)\"", TraditionalForm], 
    FormBox["\"f(d) or g(d)\"", TraditionalForm]},
  AxesOrigin->{0, 0},
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.402209385969205*^9, 3.402211289085752*^9, 
  3.4022176666462336`*^9, 3.402217715296189*^9, 3.4022179572040353`*^9, 
  3.430481812944264*^9}]
}, Open  ]],

Cell[TextData[{
 "Here we plot the distributions of smallest dimensions from one side of the \
network to another for networks with ",
 Cell[BoxData[
  FormBox[
   OverscriptBox["c", "_"], TraditionalForm]]],
 " = 10 and \[Epsilon] = 0.5 (solid line) and \[Epsilon] = 0.8 (dashed \
line)."
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.412414423341077*^9, 3.412414510947048*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", " ", "=", " ", 
   RowBox[{"cbar", "/", 
    RowBox[{"Log", "[", 
     RowBox[{"1", "/", "\[Epsilon]"}], "]"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"pd", " ", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", " ", "-", " ", 
       RowBox[{"Gdn", "[", 
        RowBox[{"n", "-", "1"}], "]"}]}], ")"}], 
     RowBox[{"fdn", "[", "n", "]"}]}], " ", "+", " ", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", "-", 
       RowBox[{"Fdn", "[", "n", "]"}]}], ")"}], 
     RowBox[{"gdn", "[", 
      RowBox[{"n", "-", "1"}], "]"}]}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"tbar", " ", "=", " ", 
   RowBox[{"\[Omega]bar", " ", "=", "1"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"cbar", " ", "=", " ", "10"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"pd", "/.", 
      RowBox[{"\[Epsilon]", "\[Rule]", ".5"}]}], ",", 
     RowBox[{"pd", "/.", 
      RowBox[{"\[Epsilon]", "\[Rule]", ".8"}]}]}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"d", ",", " ", "0", ",", " ", ".6"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "}"}], ",", "Dashed"}], "}"}]}]}], "]"}]}], "Input",
 CellChangeTimes->{{3.40213908855649*^9, 3.4021390891773825`*^9}, {
   3.4021391407916*^9, 3.4021391478217087`*^9}, 3.4021396545203056`*^9, {
   3.402139881556768*^9, 3.402139965377296*^9}, {3.402140000678056*^9, 
   3.4021400578502655`*^9}, {3.4021401423417587`*^9, 3.4021401497424*^9}, {
   3.4022056786082816`*^9, 3.4022056982064624`*^9}, {3.4022070915299635`*^9, 
   3.402207110156747*^9}, {3.4022071836324*^9, 3.402207253613027*^9}, {
   3.4022072937206993`*^9, 3.4022072942815056`*^9}, 3.402207346536645*^9, {
   3.4022074465204144`*^9, 3.4022074671400642`*^9}, {3.4022075232307186`*^9, 
   3.402207540245184*^9}, {3.4022112117445407`*^9, 3.4022112162910786`*^9}, 
   3.402211255537512*^9}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwV03k8lNsfB3DaQ93IHTlXI2LSao2U6XtyqVSKubiiUGNKqEGYkixFJrci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     "]]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], Dashing[{Small, Small}], 
    LineBox[CompressedData["
1:eJwVynk81Hkcx3EhahwVMpHGb8SkViIKZfp8YwppLBN2oshYxcidaJMjKm2r
dLD8ptBQWCE5plsHUVKONCOMMc59RORaU7H72z/ej9fj+Xi8qZwwVoC8nJzc
QWL/d6pXe1OV854dqeOzS2SzGLqRrI+p6hvDSSmn2fMbhsKMBKlCijWYcc2/
S+cwlMooT9Qi2wFp+LVuCPFnz8rTn1AcIc+pNZQ7hSHjIk9ZAM0Fqh9YFZEm
MPTDpMNLmcwCXkWD5pFRDDWLjcl3KWzYI8gkpw1gqEbAik4x8Ian00wUKMFQ
TnpcO5vmA4+XZv9r1I0hK+pIgxyZA40mDZ/Z7RhSbt+eUkwJhCpySP7AMwzF
1bpWXs0Lgkg1UqzhYwxNlARITxkEg0K/ehNbgCFRyiXEooVChllhFb8UQ+68
iPh57UgwHecFtGRjyJBZv+02JRbmtLXJiyMw5PPyZxavNxbCE5YtEQZjKMvm
U1B63gk4F+S569ZhDKnSxrN+MzgJT6P4p6wOYGh6YdU/LrR4ePtaX9faAUMc
dk3nd+3TgB4YhzXpYciCRMosoKQC9k5QOJCmj663Ti92N8gGoULPWstCPfQh
oIpbV8IHsYKFh+0KXbTR1G1AUFgEmWdkk48qyGh2g0ZCslE5VFbEBEa4rkQ7
c0ksS/we9DhX/8TarYmQPWXmirQaTGOn43x3rkAje9dSc1Y/gPGlhosP/q2O
Jv2cNLaueQw5s6+qzLtVkOXlRVoPfWvh8Pwz++DuJWgsICQkYMtzCDUXlMa7
KyGtGtZKRuULGMlJ0KzWUkRszaT6LLc6yEucODQvWYSYsyCy+FAPXHpBNEdZ
DvkMOUwxYhqgM0X/7FzUD2Ccb2Td0XgN9Je0gvNOMuiq6NTyuvIG7J4XMGT8
Wbg4bNF7avNbuEbZGI+WTcMkn6I6UtMM8Vp3eYyRrxAZzNrZ6P0eWqoLVUpu
jYN/VMLBhLgWQBvfdMnLj8Lg1I0mOr0VyvyaG5plIzDzTUK3Um8Dpa2hp7ap
DIGYcbRvX2sbZLUY6Wzn9YP5iEUhL70dGLk6n1bH9oGGBhZBOfQBBprc18+8
E8PbePejddQOuGMm1s726wKtyNLgW1MdIOV/GyetFMGrp3ZpFc8/wvyRGM9h
qw5wc8EkkiQhjKZk30kWtQH3VwMrZ5YIytroZyjpLeD95GitSKsTmtRMOsv2
N4MgbCrocnsnXFxWbGuj/gZuFuWsOsP/BIZVe3cMSV+Bv+GVOZ5/F4yZfur6
yKiDStpf3WOW3fC+TvnwpRfPoTuYNMmZ64aLgzo+apxaCCMVrle43wOUIjRX
0fgI0jcFBojPiSHcIVaa4XEffv8zoq3bqRf0aIULLvHVQHr5nqqiLoHxUa99
2Q33QHE4JZgeLYEZYcsGnYi7wDURM5M6JJB0wkY+amkpkKud7knpfeA+7KD4
8EAxmPnuL/fI7YPjvEg9VfZtWNhyji5e6IOx2yklug/zQdO1gZlyWAr+VUpB
MpubkHGhZHZ7gxTqpblVe/bngA+JT11h1g8mBet7BOE4MBXeFa8hnJiQT3Mk
bPtDZr6BcM/pzxGiMBx0v7jZ2ROuHxhUk4XiIGxd5B9N2M4x2XtbCA6u2X4F
nYSlpl7MJ0E42BlT1/HN+0F4dteFOg4O5lRmeRnhZofj6h6EMd0TVo8Iv3A4
dnXQD4cFlVaHD4SVR4R8JcKPxhMDlTb3A0VsK3T0xcFSICnmEs60j8ls9sZh
7V3VzTGEp9Nit/oQ1ii2fphM2DB/t+iLFw5f8fQ31wnPL3eiLidcFo8+vyM8
ZDrUyGLjcCPmaFQX4QRFZkz/Lzj8EZ71fZhwJTdx3THCXM6EyiKLfuC93ZuW
4YnDfm+9a2qEIx8M2tEIO7o76ukS1td1/lbjgYM181gBjbDicFylA+F1u/NM
LAhb6pwMFbnj8B8BUbRJ
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.402209388703136*^9, 3.402211292510677*^9, 
  3.4022176691498337`*^9, 3.4022177207139792`*^9, 3.4022179599379663`*^9}]
}, Open  ]],

Cell["\<\
The fraction of the distribution drawn from the pore height distribution is \
plotted against porosity in the following graphic. Lines represent networks \
with mean coverage 5 (dashed), 10 (solid) and 100 (dotted):\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.412414536143278*^9, 3.4124146332128572`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"Clear", "[", "cbar", "]"}], "\n", 
 RowBox[{
  RowBox[{"Phcbar5", " ", "=", " ", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\[Epsilon]", ",", " ", 
       RowBox[{"NIntegrate", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{"1", "-", 
              RowBox[{"Fdn", "[", "n", "]"}]}], ")"}], 
            RowBox[{"gdn", "[", 
             RowBox[{"n", "-", "1"}], "]"}]}], ")"}], "/.", 
          RowBox[{"cbar", "\[Rule]", "5"}]}], ",", " ", 
         RowBox[{"{", 
          RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], 
        "]"}]}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
      "\[Epsilon]", ",", " ", ".2", ",", " ", ".95", ",", " ", ".01"}], 
      "}"}]}], "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Phcbar10", " ", "=", " ", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\[Epsilon]", ",", " ", 
       RowBox[{"NIntegrate", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{"1", "-", 
              RowBox[{"Fdn", "[", "n", "]"}]}], ")"}], 
            RowBox[{"gdn", "[", 
             RowBox[{"n", "-", "1"}], "]"}]}], ")"}], "/.", 
          RowBox[{"cbar", "\[Rule]", "10"}]}], ",", " ", 
         RowBox[{"{", 
          RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], 
        "]"}]}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
      "\[Epsilon]", ",", " ", ".2", ",", " ", ".95", ",", " ", ".01"}], 
      "}"}]}], "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Phcbar100", " ", "=", " ", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\[Epsilon]", ",", " ", 
       RowBox[{"NIntegrate", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{"1", "-", 
              RowBox[{"Fdn", "[", "n", "]"}]}], ")"}], 
            RowBox[{"gdn", "[", 
             RowBox[{"n", "-", "1"}], "]"}]}], ")"}], "/.", 
          RowBox[{"cbar", "\[Rule]", "100"}]}], ",", " ", 
         RowBox[{"{", 
          RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], 
        "]"}]}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
      "\[Epsilon]", ",", " ", ".2", ",", " ", ".95", ",", " ", ".01"}], 
      "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"Phcbar10", ",", "Phcbar5", ",", "Phcbar100"}], "}"}], ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "}"}], ",", "Dashed", ",", " ", "Dotted"}], "}"}]}], ",", 
   " ", 
   RowBox[{"PlotRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], ",", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<\[Epsilon]\>\"", ",", "\"\<P(h)\>\""}], "}"}]}], ",", " ", 
   RowBox[{"GridLines", "\[Rule]", "Automatic"}]}], "]"}]}], "Input",
 CellChangeTimes->{{3.402208852632304*^9, 3.4022088563476467`*^9}, {
   3.402209152022806*^9, 3.4022091560485954`*^9}, 3.4022092146128063`*^9, 
   3.4022093267240143`*^9}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwt0n1M1HUcB/CjZRKxSsZtrgLKJU8jlUmIELybJhii7kiYJFIXomIG9MBC
YLNUQJzXYT6UiMOhyXiIcEScpJa4Q9HD4XHACef5+93D7x5+DxtO3CB13e/4
fLbb/fX7fD+f9+vzzhel2UUvKBSKLN9P/m88LdcQZms6awsPjsNWGdqVwd7G
H6Y4YVvHBGLytQMz0QZsTWDL6+6a8XVq0MT5smGoj+V+3Cjcx+XwGl6lu4tI
r06zZcEUFP4aQUDR6t0/LrbgQpn8wQgmRy8EPln6AOtZ+YF7uLN0lb7hPSsE
lc7X4R6uRHIfJS1/iAbfa/laI0I+ubrtKyWDlf4axY3EMk1iAwPf476Goyi8
qE2fWciiSh630oTzOcljbdUsIuRxwsdwq+fgf7kSixtyu4ExPFaZ2wK227Br
p1zjyI9qv9Z9y4ag+YZAzJI7e+PtkLuFdk0gzXrCgl/syJbHU5lx9KWKzPfn
7Hgit5sxQ1j26+G8Tx2Yz+8+vgvZ9OxynwNp/oaTWNGssuwIcVKek5g1XVUU
73HCP17NFHSBwaNj/zgpXwtam3bF/B3CYdgglwUlXyYvjynkKO8H0A5Vb1px
iYPSP6AVsYsr3mSfcpS/FfqLhjVZ6S742+U/xOvpP+07oHGRB4OjpzaWnjC6
aB8GeXru+s9KN/kwKH1l+pEm1037MXBOdRacOukmLxbfpKgSrhjdtC+LS8ln
Sl971UN+LPqfHxptzvDQ/jZ4ooK56v0e8rRhwbVpobfXg/l2NhiUn/N5Xg/5
2tFTXHTu+zAv5WNHx9sJtWGbveRthxDwljFvv5fycqCFiS9P7vKSvwMFrWfD
hqe8lJ8D+96tT4oI5OkenMg60/k4dSVPeToxPFuxLnE7T/fhRNp1Gxday1O+
HOLWm9Tc7zzdCwdrZPaiPhNPeXPoO6yOPj7H0/24sPfFQV1thED5u6Ae7+k/
u1age3IhLbtJ6dgpkIcbsX+tay+qF8jDjZaXSwqWdQjk4UbKoQ2xmQaBPNzQ
9OdE6wWBPDwIHLG3dgSL5OFBgWbr2qA4kTw8UJUY58RMkTy8SHravmVHsUge
Xmz4ob6mqk6k+/TiZnhVy5rfRPLgEf7hAWZoQCQPHoN1Wm4JI5IHD0dK5Gr1
M5E8BDSwJ283viGRhwBmuu1f9yqJPAQ03Rw8sjtHIg8RR8zx5anfSuQh4oNF
zxcePyaRh4jmjD/7z3VL5CHhsz29Ed0jEnlIqGQ266MeSfgfnZBtXg==
     "]]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], Dashing[{Small, Small}], 
    LineBox[CompressedData["
1:eJwt0ntMU3cUB/COZRGLbnE6dBl0gps8BBF1mFXkG2EOVCQCnUbXDYfMCQmT
Zc1IAEdsZMKEUIfIMAXMMOoygbgJwgIGCyaDFtcWKK9a+rh93ts6Z1iGJmO9
5Zyk6V/3/M75fk5E/pmcz4MEAkGm/8f/X23mawTx0btf268dg7VsXWe6ZRRN
r8ZU6B/8iRhpvWohWoMTyY2Fd2e0+GqPcKq9ZAx7ty0OvanXoU9UxWb3PkLc
TxVN3SN6CAKlReU50Udxg+O4XsJ/oMXI/vCE2O4JZFj4B3SICB7uyb81CS67
199Bh4mO4LOKZgMU/tek9XocXvNbt6x6CjsCNY7EFLfRIJuG/3F/w3GMdoXm
nf90BuX8uGUTUG3dmXZ43yze5scRTaLzfuXU6zFzGOLbqSahjOw78fNKI744
xZcBuySL6gWHEcLlhnhiEJ80DT4G321d5xRuam+cP9RkQg4/XvY0VhnuSROL
5vEP325hGiWWNt1AmBnL+c3g8uqODoXcjJRAw1lkpUYfrHOYKc9ZRNb0hd3J
sCAwXtUcuJuZzPNbFsrXiJWflTcWrbBiTMOXES8fE6a+VGClvB9DnbRlafC+
FW8EBjQh73SF9vZ6G+VvQnxXdcMfX9oQaCedh0zcwm0YtpGHGaYa78729Qzt
Y8Z7z14UFhcy5GPGE9ke0bd9DO1nxr8/FhQZV9jJy4Kk39Nt9RI77WvBQ5/2
wrU2O/lZUPSfvD/Ubaf9rbi+JREvtjnI04rE4nvhR0sdWG5nhU75TteOfgf5
2pCcq6+9uuSgfGyISrpd27jXSd42nMKZ/q1yJ+XFIPMv8yclD5zkz+BAi4Ap
WHJSfgwylLuehia76B7s+FCsOHKp1EV52vHxmqW183dcdB92NKofHQthXZSv
A8NDaRcjNrnpXhyQ1DWkxh93U94OvFUrOSJWuOl+nHhuaTBLHropfyeCftGk
1yy66Z6c4JpDDjBxHvJwIfZGq/DrPA95uDAapY5Mu+QhDxfK26J68lUe8nCh
9ULBRfPfHvJwY+MHdbGaSJY83JBx4UFJOSx5uPH+lYHLCedY8vAgK2j1r+ou
ljw8UP5gHBCZWLpPD9LVJ4VYxZEHi7CnCwf3iTnyYLG5WJ5w6DRHHiy+L6uO
+OYKRx4cfK2vHGeGOfLgkJL1XZbqGUceHOS7K++mbvKShxc+6bWz7ble8vAi
ZDuTur3KSx5elAYfRW6vlzx8iO6pm6v1esnDB03Lho3Kd334H9UdZOo=
     "]]}, 
   {Hue[0.14213595499957954`, 0.6, 0.6], Dashing[{0, Small}], 
    LineBox[CompressedData["
1:eJwt0mtQlGUUB/BNGmXWyQnRiqG2YSYCRylGmUQa/BvkbCLEzZpWUG4tfNgG
iU2qFXTCCGlGcU0g7YMitn4QNiIrmAFHNz8gF3U1Lq6w7S573/d9ni4CDQG2
73LOzM5+es9zzv93YkoP5alXyWSyzOBP+j9/Tqrb6NuynFE+5IRDt8GotA9h
urq2K7rChU2FzabZ+BEIiqTIiwsufJwqn+ioGsW+8iuKygY3+hQNgdzeO7j5
UfrQ8hoPZKG6h5J66/uD9R5crpI+uIfaibS61DkP3rFLD5gxm5UzMVLqhZDb
G+xgRsn6RzW2QS9OB18rbL4P/YeW9ul4H7aF6gHm0653vnLch+DjwYYPcKPW
Kjw76cMRaVzd74jdLQ78G+fHy9I4ijH4/05oTdD68ZvUzjSGWzpf7GKfHxXl
Uo3DcDL5WOeiH/KVhvh0kym9/s0ApG4bjBOw9xwb768JIE8aL3cSU0/mm9uM
AcxJ7WYn8bz2p3/2zASwkt9DFF9irjUbBewMNbTg7P1B21NvC5SnBcYOVby6
SkBovIZHiFmLpYLzAuU7hcUj+aPPmASMjkg1hbgn+0oveATKexqP51Trtq0V
sTE0oBVfbJWNe7eIlL8VtzXZ31ozRYTaFf6B/yLykhI1InnYcOJSZed8o0j7
2NBTqVdmdYjkYwP7JaVbOSDSfjbUTjVGLI6J5GVHcmbO1qOiSPvaUTx96JQt
jJGfHWccH1QmRTHa34ED28OXmhIYeTrwwluHC//cxbDSzoG7uW29h/MZ+c5g
7LkoVZyaUT4zGHaYf4yoYeQ9g7Zr0dW7vmKUlxNKd/Q311sY+TuhORqpbr3M
KD8n2pOGvzP3MLoHF7KLX9v/+Q1GebqwKisutXWU0X24EDV8R51oYZSvG5Eq
2fFsN6N7cUMnllkW/mKUtxtFRcs5O5YY3Y8HP5xQaSPCOeXvQcvd9+qa1nO6
Jw+udv1acuVFTh5e5Kc3mKte5eThxcHUjPTA65w8vBjRFixE7+Dk4YUiLHld
eBonDx/OPdxcfS2Dk4cPAynFS7H5nDx8kJnOaEsKOHn4wT/rP6Up4+ThR5FL
Lt+j4XSffnx5sNu4XM3JI4C6l3aX6XWcPALo2ntz5+p6Th4BRI3rL5Y2cfIQ
sD1DntN5mpOHAENLt8HXxslDgP6Nx0MxFzh5iGjvOfn0AQMnDxH7G5tTvu/i
5CEiUfl17+qfOXkwhMXu/UTfz8mDwVlheLfgFsf/B6Vulw==
     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox["\"\[Epsilon]\"", TraditionalForm], 
    FormBox["\"P(h)\"", TraditionalForm]},
  AxesOrigin->{0, 0},
  GridLines->Automatic,
  PlotRange->{{0, 1}, {0, 1}},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.402209392839083*^9, 3.402211296716725*^9, 
  3.4022176732557373`*^9, 3.4022177245795374`*^9, 3.4022179645145473`*^9}]
}, Open  ]],

Cell[TextData[{
 "Void structure can be influence by using ribbon-like fibres with \
cross-sectional aspect ratio ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["A", "xs"], TraditionalForm]]],
 " = ",
 Cell[BoxData[
  FormBox[
   OverscriptBox["\[Omega]", "_"], TraditionalForm]]],
 "/",
 Cell[BoxData[
  FormBox[
   OverscriptBox["t", "_"], TraditionalForm]]],
 ". Here we plot the fraction of the distribution drawn from the pore height \
distribution against  this parameter for fibres of unit width for networks \
with \[Epsilon] = 0.5 (solid line), 0.8 (dashed) and 0.9 (dotted):"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.4124146501572227`*^9, 3.4124148461690736`*^9}, {
   3.4124149116932926`*^9, 3.412414912975136*^9}, 3.412415827470115*^9}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"Clear", "[", 
  RowBox[{"tbar", ",", " ", "\[Omega]bar"}], "]"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"tbar", " ", "=", " ", 
   RowBox[{"\[Omega]bar", "/", "Axs"}]}], ";", " ", 
  RowBox[{"\[Omega]bar", " ", "=", " ", "1"}], ";", 
  RowBox[{"cbar", " ", "=", " ", "10"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Ph\[Epsilon]5", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"Axs", ",", " ", 
       RowBox[{"NIntegrate", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{"1", "-", 
              RowBox[{"Fdn", "[", "n", "]"}]}], ")"}], 
            RowBox[{"gdn", "[", 
             RowBox[{"n", "-", "1"}], "]"}]}], ")"}], "/.", 
          RowBox[{"\[Epsilon]", "\[Rule]", "0.5"}]}], ",", " ", 
         RowBox[{"{", 
          RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], 
        "]"}]}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"Axs", ",", " ", "1", ",", " ", "10", ",", " ", ".2"}], "}"}]}],
     "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Ph\[Epsilon]8", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"Axs", ",", " ", 
       RowBox[{"NIntegrate", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{"1", "-", 
              RowBox[{"Fdn", "[", "n", "]"}]}], ")"}], 
            RowBox[{"gdn", "[", 
             RowBox[{"n", "-", "1"}], "]"}]}], ")"}], "/.", 
          RowBox[{"\[Epsilon]", "\[Rule]", "0.8"}]}], ",", " ", 
         RowBox[{"{", 
          RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], 
        "]"}]}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"Axs", ",", " ", "1", ",", " ", "10", ",", " ", ".2"}], "}"}]}],
     "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"Ph\[Epsilon]9", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"Axs", ",", " ", 
       RowBox[{"NIntegrate", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{"1", "-", 
              RowBox[{"Fdn", "[", "n", "]"}]}], ")"}], 
            RowBox[{"gdn", "[", 
             RowBox[{"n", "-", "1"}], "]"}]}], ")"}], "/.", 
          RowBox[{"\[Epsilon]", "\[Rule]", "0.9"}]}], ",", " ", 
         RowBox[{"{", 
          RowBox[{"d", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}]}], 
        "]"}]}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"Axs", ",", " ", "1", ",", " ", "10", ",", " ", ".2"}], "}"}]}],
     "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"Ph\[Epsilon]5", ",", "Ph\[Epsilon]8", ",", "Ph\[Epsilon]9"}], 
    "}"}], ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "}"}], ",", "Dashed", ",", " ", "Dotted"}], "}"}]}], ",", 
   " ", 
   RowBox[{"PlotRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "10"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"0.5", ",", "1"}], "}"}]}], "}"}]}], ",", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
     "\"\<\!\(\*SubscriptBox[\(A\), \
\(xs\)]\)=\!\(\*OverscriptBox[\(\[Omega]\), \
\(_\)]\)/\!\(\*OverscriptBox[\(t\), \(_\)]\)\>\"", ",", "\"\<P(h)\>\""}], 
     "}"}]}], ",", " ", 
   RowBox[{"GridLines", "\[Rule]", "Automatic"}]}], 
  "]"}], "\[IndentingNewLine]", 
 RowBox[{"Clear", "[", 
  RowBox[{"tbar", ",", "\[Omega]bar"}], "]"}]}], "Input",
 CellChangeTimes->{{3.4022115211294146`*^9, 3.4022115949054995`*^9}, {
  3.4022116322792397`*^9, 3.402211926922917*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    LineBox[{{1., 0.7444555448425529}, {1.2, 0.7877928195928854}, {1.4, 
     0.8207180514628223}, {1.6, 0.8463258183823296}, {1.8, 
     0.8666439527046373}, {2., 0.883044506636848}, {2.2, 
     0.8964819339503569}, {2.4, 0.9076364888493205}, {2.6, 
     0.9170034442962431}, {2.8000000000000003`, 0.9249502882012186}, {3., 
     0.9317543330927902}, {3.2, 0.9376280366146451}, {3.4000000000000004`, 
     0.9427364069861546}, {3.6, 0.9472091954685548}, {3.8000000000000003`, 
     0.9511495845894029}, {4., 0.9546404765918294}, {4.2, 
     0.9577491103446913}, {4.4, 0.9605304957077204}, {4.6000000000000005`, 
     0.9630299992396458}, {4.8, 0.9652853127715325}, {5., 
     0.9673279676895554}, {5.2, 0.9691845109884318}, {5.4, 
     0.9708774268353607}, {5.6000000000000005`, 0.9724258647587904}, {
     5.800000000000001, 0.9738462195427529}, {6., 0.975152596414355}, {6.2, 
     0.9763571867845914}, {6.4, 0.9774705737078685}, {6.6000000000000005`, 
     0.9785019817223728}, {6.800000000000001, 0.9794594823763153}, {7., 
     0.9803501642211525}, {7.2, 0.9811802741402563}, {7.4, 
     0.9819553354210622}, {7.6000000000000005`, 0.9826802468556426}, {
     7.800000000000001, 0.9833593662850427}, {8., 0.9839965813250862}, {
     8.200000000000001, 0.984595369480011}, {8.4, 0.9851588494312562}, {8.6, 
     0.9856898249563832}, {8.8, 0.9861908226681186}, {9., 
     0.9866641245511697}, {9.200000000000001, 0.9871117961034354}, {9.4, 
     0.9875357107498959}, {9.6, 0.9879375710850066}, {9.8, 
     0.9883189274076597}, {10., 0.9886811939375585}}]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], Dashing[{Small, Small}], 
    LineBox[{{1., 0.9116259875631094}, {1.2, 0.9312131998704807}, {1.4, 
     0.9447165689944883}, {1.6, 0.9544447858933943}, {1.8, 
     0.9617028464195974}, {2., 0.9672740912806261}, {2.2, 
     0.9716523011835375}, {2.4, 0.9751616102654266}, {2.6, 
     0.9780221770200112}, {2.8000000000000003`, 0.9803879530517535}, {3., 
     0.9823693290695328}, {3.2, 0.9840472097459158}, {3.4000000000000004`, 
     0.9854820391651421}, {3.6, 0.9867197493644506}, {3.8000000000000003`, 
     0.9877957772957788}, {4., 0.9887378367500115}, {4.2, 
     0.9895678686031886}, {4.4, 0.9903034371606813}, {4.6000000000000005`, 
     0.9909587458842917}, {4.8, 0.9915453842137222}, {5., 
     0.9920728990701869}, {5.2, 0.9925492028118438}, {5.4, 
     0.9929809086899315}, {5.6000000000000005`, 0.9933735792687071}, {
     5.800000000000001, 0.9937319224361223}, {6., 0.9940599457835927}, {6.2, 
     0.9943610792035229}, {6.4, 0.994638272999915}, {6.6000000000000005`, 
     0.9948940769705329}, {6.800000000000001, 0.9951307045831657}, {7., 
     0.9953501073216083}, {7.2, 0.9955539080730941}, {7.4, 
     0.995743656049937}, {7.6000000000000005`, 0.9959206369954438}, {
     7.800000000000001, 0.9960860075258341}, {8., 0.9962407938885981}, {
     8.200000000000001, 0.9963859093984957}, {8.4, 0.9965221691925508}, {8.6, 
     0.9966503027674664}, {8.8, 0.9967709646742999}, {9., 
     0.9968847436751805}, {9.200000000000001, 0.9969921706110743}, {9.4, 
     0.9970937251849651}, {9.6, 0.9971898418289422}, {9.8, 
     0.9972809147946695}, {10., 0.9973673025831943}}]}, 
   {Hue[0.14213595499957954`, 0.6, 0.6], Dashing[{0, Small}], 
    LineBox[{{1., 0.9479258261182966}, {1.2, 0.9601502440681294}, {1.4, 
     0.9683591708510192}, {1.6, 0.9741604736834923}, {1.8, 
     0.9784262332154008}, {2., 0.9816638163221689}, {2.2, 
     0.9841853625263242}, {2.4, 0.986191848270594}, {2.6, 
     0.9878176597574694}, {2.8000000000000003`, 0.9891555662882201}, {3., 
     0.990271375911756}, {3.2, 0.9912128824797838}, {3.4000000000000004`, 
     0.9920155168543013}, {3.6, 0.9927060238268923}, {3.8000000000000003`, 
     0.9933049176748198}, {4., 0.993828160371688}, {4.2, 
     0.9942883324096441}, {4.4, 0.9946954648822381}, {4.6000000000000005`, 
     0.9950576407835204}, {4.8, 0.9953814361609064}, {5., 
     0.9956722482694483}, {5.2, 0.9959345427715242}, {5.4, 
     0.9961720421239345}, {5.6000000000000005`, 0.9963878706864879}, {
     5.800000000000001, 0.9965846676037279}, {6., 0.9967646754247281}, {6.2, 
     0.9969298102704394}, {6.4, 0.9970817178334872}, {6.6000000000000005`, 
     0.9972218184035605}, {6.800000000000001, 0.9973513433214131}, {7., 
     0.9974713646861124}, {7.2, 0.9975828172962031}, {7.4, 
     0.9976865283433483}, {7.6000000000000005`, 0.9977832197511319}, {
     7.800000000000001, 0.9978735315407828}, {8., 0.9979580311026016}, {
     8.200000000000001, 0.9980372229789332}, {8.4, 0.9981115571318029}, {8.6, 
     0.9981814359591425}, {8.8, 0.9982472231653735}, {9., 
     0.9983092373606965}, {9.200000000000001, 0.9983677736323873}, {9.4, 
     0.9984230959270357}, {9.6, 0.9984754431511974}, {9.8, 
     0.9985250320025355}, {10., 0.9985720594313812}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox[
    "\"\\!\\(\\*SubscriptBox[\\(A\\), \\(xs\\)]\\)=\\!\\(\\*OverscriptBox[\\(\
\[Omega]\\), \\(_\\)]\\)/\\!\\(\\*OverscriptBox[\\(t\\), \\(_\\)]\\)\"", 
     TraditionalForm], 
    FormBox["\"P(h)\"", TraditionalForm]},
  AxesOrigin->{0, 0.5},
  GridLines->Automatic,
  PlotRange->{{0, 10}, {0.5, 1}},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.402212077249075*^9, 3.4022176757092657`*^9, 
  3.402217727073123*^9, 3.4022179670582047`*^9}]
}, Open  ]],

Cell["\<\
The influence of fibre cross-sectional aspect ratio on the mean pore diameter \
is best illustrated by considering fibres of constant cross-sectional area:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.412414855172019*^9, 3.412414905274062*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"Clear", "[", 
  RowBox[{"cbar", ",", "\[Epsilon]"}], "]"}], "\[IndentingNewLine]", 
 RowBox[{" ", 
  RowBox[{
   RowBox[{"\[Omega]bar", " ", "=", " ", "1"}], ";", 
   RowBox[{"tbar", " ", "=", " ", 
    RowBox[{"\[Omega]bar", "/", 
     SqrtBox["Axs"]}]}], ";", " ", 
   RowBox[{"cbar", " ", "=", " ", "10"}], ";"}], 
  " "}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Epsilon]", "=", ".8"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"dbarAxs", " ", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"Axs", ",", 
       RowBox[{"NIntegrate", "[", 
        RowBox[{
         RowBox[{"d", " ", "pd"}], ",", " ", 
         RowBox[{"{", 
          RowBox[{"d", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]}], 
      "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"Axs", ",", " ", "1", ",", " ", "10", ",", ".5"}], "}"}]}], 
    "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"dbarAxs", ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", " ", "\[IndentingNewLine]", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
     "\"\<\!\(\*SubscriptBox[\(A\), \
\(xs\)]\)=\!\(\*OverscriptBox[\(\[Omega]\), \
\(_\)]\)/\!\(\*OverscriptBox[\(t\), \(_\)]\)\>\"", ",", 
      "\"\<\!\(\*OverscriptBox[\(d\), \(_\)]\)/\!\(\*OverscriptBox[\(\[Omega]\
\), \(_\)]\)\>\""}], "}"}]}]}], "]"}]}], "Input",
 CellChangeTimes->{{3.402212352444787*^9, 3.402212568054819*^9}, {
   3.4022126065902305`*^9, 3.4022127792284718`*^9}, {3.4022134574336834`*^9, 
   3.4022135154270735`*^9}, {3.4022135467721453`*^9, 3.4022136139687696`*^9}, 
   3.4022136473667936`*^9, {3.4022137136420927`*^9, 3.402213805273853*^9}, {
   3.402213845762072*^9, 3.40221385222136*^9}, {3.402213889655187*^9, 
   3.402214052038683*^9}, {3.4022141563086157`*^9, 3.4022141671341825`*^9}, {
   3.4022143209353375`*^9, 3.4022143936398816`*^9}, {3.4022144646319633`*^9, 
   3.4022145155151296`*^9}, {3.402214560720131*^9, 3.402214564685834*^9}, {
   3.4022146342158127`*^9, 3.4022146842577696`*^9}, {3.4022148116910095`*^9, 
   3.402214869884688*^9}, {3.402216466830984*^9, 3.402216493990037*^9}, {
   3.4022165670350704`*^9, 3.4022166090254498`*^9}, {3.402216741075328*^9, 
   3.4022169156363344`*^9}, {3.40221749646152*^9, 3.402217576406475*^9}, {
   3.402217613780216*^9, 3.402217625877611*^9}, {3.4022176904003906`*^9, 
   3.402217690540592*^9}, {3.402217842909688*^9, 3.4022179386373377`*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], 
    LineBox[{{1., 0.10747374737299961`}, {1.5, 0.08982411969920895}, {2., 
     0.07882032365044166}, {2.5, 0.07110562166479797}, {3., 
     0.065305675523506}, {3.5, 0.06073737462267019}, {4., 
     0.057017116992464756`}, {4.5, 0.05391045310554962}, {5., 
     0.05126484936754484}, {5.5, 0.048976170486881375`}, {6., 
     0.046970546899775936`}, {6.5, 0.04519391595164601}, {7., 
     0.04360567087827029}, {7.5, 0.04217463502404311}, {8., 
     0.040876418081336185`}, {8.5, 0.03969162586547458}, {9., 
     0.03860461582902369}, {9.5, 0.037602611892699256`}, {10., 
     0.03667506202087081}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox[
    "\"\\!\\(\\*SubscriptBox[\\(A\\), \\(xs\\)]\\)=\\!\\(\\*OverscriptBox[\\(\
\[Omega]\\), \\(_\\)]\\)/\\!\\(\\*OverscriptBox[\\(t\\), \\(_\\)]\\)\"", 
     TraditionalForm], 
    FormBox[
    "\"\\!\\(\\*OverscriptBox[\\(d\\), \\(_\\)]\\)/\\!\\(\\*OverscriptBox[\\(\
\[Omega]\\), \\(_\\)]\\)\"", TraditionalForm]},
  AxesOrigin->{2., 0.04},
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{
  3.4022166124103165`*^9, {3.4022167711786146`*^9, 3.40221677883963*^9}, {
   3.4022168504225616`*^9, 3.402216868939187*^9}, {3.40221690310832*^9, 
   3.40221691687812*^9}, 3.4022175191241074`*^9, 3.4022175775781603`*^9, 
   3.4022176286916575`*^9, {3.4022176771112814`*^9, 3.4022176913217154`*^9}, 
   3.402217727623915*^9, {3.4022178449225826`*^9, 3.4022178517724323`*^9}, 
   3.402217885651147*^9, {3.4022179173066654`*^9, 3.4022179675889683`*^9}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["5.5 Tortuosity", "Section",
 Editable->False,
 CellChangeTimes->{{3.4022245899013853`*^9, 3.402224595799867*^9}, {
  3.4124149276963043`*^9, 3.4124149303100624`*^9}}],

Cell["\<\
For isotropically porous networks, the tortuosity is obtained as:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.412414950449021*^9, 3.4124150058186383`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"Sum", "[", 
   RowBox[{
    RowBox[{"i", " ", "\[Epsilon]", " ", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", " ", "-", " ", "\[Epsilon]"}], ")"}], 
      RowBox[{"i", "-", "1"}]]}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"i", ",", " ", "1", ",", " ", "\[Infinity]"}], "}"}]}], "]"}], 
  "]"}]], "Input",
 CellChangeTimes->{{3.4023911827301087`*^9, 3.4023912642573395`*^9}}],

Cell[BoxData[
 FractionBox["1", "\[Epsilon]"]], "Output",
 CellChangeTimes->{{3.4023912072253313`*^9, 3.4023912647380304`*^9}, 
   3.402391305686912*^9}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["5.6 Distributions of Thickness and Mass", "Section",
 Editable->False,
 CellChangeTimes->{{3.4027299854149666`*^9, 3.4027299943778543`*^9}, {
  3.412415055650293*^9, 3.4124150564113874`*^9}}],

Cell[TextData[{
 "We consider first a network of mean areal density ",
 Cell[BoxData[
  FormBox[
   OverscriptBox["\[Beta]", "_"], TraditionalForm]]],
 " = 60 g ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    StyleBox["m",
     FontSlant->"Plain"], 
    RowBox[{"-", "2"}]], TraditionalForm]]],
 "and mean porosity \[Epsilon] = 0.8 formed from fibres with density 1.5 g ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["cm", 
    RowBox[{"-", "3"}]], TraditionalForm]]],
 "; these properties define the mean thickness, ",
 Cell[BoxData[
  FormBox[
   OverscriptBox["z", "_"], TraditionalForm]]],
 ". The coefficient of variation of thickness and areal density are 0.1 and \
the correlation between thickness and density is 0.6:"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.412415189102187*^9, 3.4124153192693586`*^9}, {
  3.412415364093813*^9, 3.4124154017379427`*^9}}],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"\[Beta]bar", " ", "=", " ", "60"}], ";"}], " ", 
  RowBox[{"(*", " ", 
   RowBox[{"g", " ", 
    SuperscriptBox["m", 
     RowBox[{"-", "2"}]]}], " ", "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"\[Rho]f", " ", "=", " ", "1.5"}], ";"}], " ", 
  RowBox[{"(*", " ", 
   RowBox[{"g", " ", 
    SuperscriptBox["cm", 
     RowBox[{"-", "3"}]]}], "*)"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Epsilon]bar", " ", "=", " ", "0.8"}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"zbar", " ", "=", " ", 
   RowBox[{"\[Beta]bar", "/", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", "\[Epsilon]bar"}], ")"}], " ", "\[Rho]f"}], 
     ")"}]}]}], " ", 
  RowBox[{"(*", " ", "\[Mu]m", " ", "*)"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"cvz", " ", "=", " ", 
   RowBox[{"cv\[Beta]", " ", "=", " ", "0.1"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Sigma]z", " ", "=", " ", 
   RowBox[{"cvz", " ", "zbar"}]}], ";", 
  RowBox[{"\[Sigma]\[Beta]", " ", "=", " ", 
   RowBox[{"cv\[Beta]", " ", "\[Beta]bar"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Rho]corr", "=", " ", ".6"}], ";"}]}], "Input",
 CellChangeTimes->{{3.4028165717199583`*^9, 3.402816576627014*^9}}],

Cell["\<\
We generate a million random pairs representing thickness and areal density \
of our network by drawing these from a bivariate normal distribution:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.412415325388157*^9, 3.412415355611616*^9}, {
  3.412415411091392*^9, 3.412415465649843*^9}}],

Cell[BoxData[{
 RowBox[{"Needs", "[", "\"\<MultivariateStatistics`\>\"", 
  "]"}], "\[IndentingNewLine]", 
 RowBox[{"SeedRandom", "[", "1", "]"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Beta]z", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{
     RowBox[{"MultinormalDistribution", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"\[Beta]bar", ",", "zbar"}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           SuperscriptBox["\[Sigma]\[Beta]", "2"], ",", 
           RowBox[{"\[Rho]corr", " ", "\[Sigma]\[Beta]", " ", "\[Sigma]z"}]}],
           "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"\[Rho]corr", " ", "\[Sigma]\[Beta]", " ", "\[Sigma]z"}], 
           ",", 
           SuperscriptBox["\[Sigma]z", "2"]}], "}"}]}], "}"}]}], "]"}], ",", 
     "1000000"}], "]"}]}], ";"}]}], "Input",
 CellChangeTimes->{{3.402816602484195*^9, 3.402816602674469*^9}, {
  3.4305455665174*^9, 3.430545570673376*^9}}],

Cell[TextData[{
 "To allow rendering of graphics, we plot a random selection of 2000 of these \
points, selected using ",
 StyleBox["RandomSample", "Input"],
 ":"
}], "Text",
 Editable->False,
 CellChangeTimes->{{3.4124154719689293`*^9, 3.4124155089721375`*^9}, {
  3.4305453892925634`*^9, 3.4305453904241905`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"RandomSample", "[", 
    RowBox[{"\[Beta]z", ",", " ", "2000"}], "]"}], ",", " ", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
     "\"\<\!\(\*OverscriptBox[\(\[Beta]\), \(~\)]\)\>\"", ",", 
      "\"\<\!\(\*OverscriptBox[\(z\), \(~\)]\)\>\""}], "}"}]}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellChangeTimes->{{3.4027302052811184`*^9, 3.4027303746045933`*^9}, {
  3.4027304162644978`*^9, 3.4027305081065598`*^9}, {3.402730604314901*^9, 
  3.4027306151004095`*^9}, {3.4027310028880205`*^9, 3.4027310031984673`*^9}, {
  3.4027346183968673`*^9, 3.4027346218518353`*^9}, {3.4027353388628464`*^9, 
  3.4027353610747857`*^9}, {3.402735394012147*^9, 3.402735405749024*^9}, {
  3.4028166164642973`*^9, 3.4028166960988064`*^9}, {3.430545373940488*^9, 
  3.4305453747616687`*^9}}],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJwUm3c41f8bxomspsqICGWkMkIkcWc7+5ilonD2sYsiRaRBJOIbUYRUSomG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     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox[
    "\"\\!\\(\\*OverscriptBox[\\(\[Beta]\\), \\(~\\)]\\)\"", TraditionalForm],
     
    FormBox[
    "\"\\!\\(\\*OverscriptBox[\\(z\\), \\(~\\)]\\)\"", TraditionalForm]},
  AxesOrigin->{40., 140.},
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.402816757697381*^9, 3.4028173278972883`*^9, 
  3.4028180881504784`*^9, 3.4028220467426543`*^9, 3.4305454983694077`*^9, 
  3.4305456185321937`*^9}]
}, Open  ]],

Cell["\<\
From the thickness and areal density, we can calculate the porosity and \
characterise its distribution:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.412415535700571*^9, 3.4124155675063057`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"\[Beta]zT", "=", 
   RowBox[{"Transpose", "[", "\[Beta]z", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"\[Epsilon]net", " ", "=", " ", 
   RowBox[{"1", "-", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"\[Beta]zT", "[", 
       RowBox[{"[", "1", "]"}], "]"}], "/", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"\[Beta]zT", "[", 
         RowBox[{"[", "2", "]"}], "]"}], " ", "\[Rho]f"}], ")"}]}], ")"}]}]}],
   ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Epsilon]std", " ", "=", " ", 
   RowBox[{"StandardDeviation", "[", "\[Epsilon]net", "]"}]}], ";"}], "\n", 
 RowBox[{"Needs", "[", "\"\<Histograms`\>\"", "]"}], "\[IndentingNewLine]", 
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"Histogram", "[", 
    RowBox[{"\[Epsilon]net", ",", 
     RowBox[{"HistogramCategories", "\[Rule]", "30"}], ",", " ", 
     RowBox[{"HistogramScale", "\[Rule]", "1"}]}], "]"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"PDF", "[", 
      RowBox[{
       RowBox[{"NormalDistribution", "[", 
        RowBox[{"\[Epsilon]bar", ",", "\[Epsilon]std"}], "]"}], ",", " ", 
       "e"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"e", ",", " ", ".7", ",", "1"}], "}"}], ",", 
     RowBox[{"PlotStyle", "\[Rule]", 
      RowBox[{"Thickness", "[", ".005", "]"}]}]}], "]"}], ",", " ", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<\[Epsilon]\>\"", ",", " ", "\"\<f(\[Epsilon])\>\""}], 
     "}"}]}]}], "]"}]}], "Input",
 CellChangeTimes->{{3.4027352096670723`*^9, 3.402735221093502*^9}, {
   3.4028158519650016`*^9, 3.402815865955118*^9}, {3.4028167042505283`*^9, 
   3.4028167264224095`*^9}, 3.402817293948472*^9, {3.430545498189149*^9, 
   3.4305455108874083`*^9}}],

Cell[BoxData[
 GraphicsBox[{{
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.675, 0}, \
{0.6799999999999999, 0.00019999999999999982`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.6800000000000002, 0}, \
{0.685, 0.00039999999999999964`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.685, 0}, {0.69, 0.00039999999999999964`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.6900000000000002, 0}, \
{0.6950000000000001, 0.00019999999999999982`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.6950000000000001, 0}, {0.7, 0.0017999999999999984`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7000000000000002, 0}, \
{0.7050000000000001, 0.0017999999999999984`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7050000000000001, 0}, {0.71, 0.005599999999999995}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7100000000000002, 0}, \
{0.7150000000000001, 0.008199999999999994}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7150000000000001, 0}, {0.72, 0.016599999999999986`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7200000000000002, 0}, \
{0.7250000000000001, 0.03159999999999997}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7250000000000001, 0}, {0.73, 0.06359999999999993}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7300000000000002, 0}, \
{0.7350000000000001, 0.12119999999999989`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7350000000000001, 0}, {0.74, 0.2089999999999998}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7400000000000002, 0}, \
{0.7450000000000001, 0.40099999999999963`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7450000000000001, 0}, {0.75, 0.6613999999999994}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7500000000000002, 0}, \
{0.7550000000000001, 1.1491999999999991`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7550000000000001, 0}, {0.76, 1.8897999999999986`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7600000000000002, 0}, \
{0.7650000000000001, 3.0167999999999973`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7650000000000001, 0}, {0.77, 4.663599999999995}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7700000000000002, 0}, \
{0.7750000000000001, 6.917799999999994}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7750000000000001, 0}, {0.78, 9.792999999999992}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7800000000000002, 0}, \
{0.7850000000000001, 13.065399999999988`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7850000000000001, 0}, {0.79, 16.624599999999987`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7900000000000003, 0}, \
{0.7950000000000002, 19.685199999999984`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.7950000000000002, 0}, {0.8, 21.722599999999982`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8000000000000003, 0}, \
{0.8050000000000002, 22.34419999999998}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8050000000000002, 0}, {0.81, 20.934199999999983`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8100000000000003, 0}, \
{0.8150000000000002, 18.215999999999983`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8150000000000002, 0}, \
{0.8200000000000001, 14.324799999999986`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8200000000000003, 0}, \
{0.8250000000000002, 10.22819999999999}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8250000000000002, 0}, \
{0.8300000000000001, 6.550399999999995}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8300000000000003, 0}, \
{0.8350000000000002, 3.824599999999997}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8350000000000002, 0}, \
{0.8400000000000001, 2.000999999999998}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8400000000000003, 0}, \
{0.8450000000000002, 0.9219999999999993}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8450000000000002, 0}, \
{0.8500000000000001, 0.38759999999999967`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8500000000000003, 0}, \
{0.8550000000000002, 0.14399999999999988`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8550000000000002, 0}, \
{0.8600000000000001, 0.051199999999999954`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8600000000000003, 0}, \
{0.8650000000000002, 0.016399999999999988`}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8650000000000002, 0}, \
{0.8700000000000001, 0.003599999999999997}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8700000000000003, 0}, \
{0.8750000000000002, 0.0005999999999999995}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8750000000000002, 0}, {0.8800000000000001, 0}]}, 
    {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
     EdgeForm[GrayLevel[0]], 
     RectangleBox[{0.8800000000000003, 0}, \
{0.8850000000000002, 0.00019999999999999982`}]}}, {{}, {}, 
    {Hue[0.67, 0.6, 0.6], Thickness[0.005], LineBox[CompressedData["
1:eJwUl3c8lu8Xx41kJhIaQpIkhGzu+9xFSNlSIdl8zeyRXUYlmZEtsirZZCWR
omEVlSLjGZ5hFFnxu39/eb1fPNd1zud8zud6HLb3NnViYmBguLqDgeH/P6nj
z/9zdiaAMmM1pRsZR+1dem2OuRCgUeVeO5n1Bzq2+NOMhLNe0/zHm6M/0J6d
3KjbfwSIkRPpqYifQPPlPPZ4eRAg88GPir+Zv9C9rTdZT3oSYKTEVn157xR6
52zO5jzOX08e+no+YwoNsuwj+HgTIPwwO//VwmnU+JZka4AvAaoeLIblTs2i
b7ixamU/Asgc6/9CCSOgmg8vPf6Lc+s+kx7zg0T0eFXc/ZAAAmQnOStOOJJQ
prEZh/BgAmTFJhbQZShosP3mZTSEAP+MKxOEqRSUTuUz3MY5VCnynslzKvqN
UUs1+gYB1AMqUrTO0tG6E0VcsREE0Cgxibv7dQGVamxm1IkkwPm4cp57hoto
ITawsjOKAMT75Pi8N4to4sXtyYRoAnCXdYN65xLqGHm1IfEWAY59/jpY+v0P
+o09oNIglgBWm2fybd2XUeP0xALuOALc6AD5ka1lFKlovZ0cTwDFj9K79OT/
ogLD+23S7+D1loQHCfeuoYlX5c0u3iWATl5Jmk3QOspE0tMTSCRATstskoT0
Bjq/GaSQdQ/vf+jTnlfFm+hbiS87c5MJMCRsCe4ft1G0hrZhnUKAJKktx31m
DFCvwbJ4KJUA0Bkm0/qKAYqMFb8XpBHAJuIiMbaMEUJDU58XP8Dv37X5XCqT
GYKOV034ZxKAY+ZtqJbEDvAfe7dbJ4sAZI1d//Re7ABPZcbrpIcE6MusglsU
FrBd8paTySMA9dsw79ubbHC16K7tFs4RvYOJhkrsYGlclvwpnwDYugC5jMIO
ZlU/F3wKCeBZ52Fh7sIJOm4GNY3FBLjdMdmvl88NWvv/+xVfQgCGCvcR/tDd
gL29xXvlMQGm9sV0EGR4QE2izWejlADWF3ievd/HC9JTUgpYJQEySkykUvj5
4HiKjj3vE1y/joh7Nnl8IIHZp07hXGpgMdUnuRdE8x8u3XqGzzc/5cyEPj/w
WbLVvavG+2t4mvK8QxB42MWns2sIMFeQ63HceR9wNQOfRy3uz5ETEbv37Ied
AkF+3PUEyD+vupAadADWB2dPmTXhepW/zuwMOwR/oxgdxZsJgJxbPvZLXxj+
nDyUvoyzRHNqr5OwCNDumf/JbCHA98+f2/e4i8Lkudf14+0E+I91O1BdRwx+
rP6cedZBAJYXg47D3WLwrWx9b+RLfB++OJXW6RyBYRaFgMOv8PldaTTPMxeH
N68KlZy7CSBF9pMJypGA19fbnFR6CJCyN8C0QvkYdIqMZbC9IcAO9s4zXz4f
gxfhu1cqewkQbXTf4ZbocXiqFtE430cA98EF811zJ4Aq189e108Apy6DUoUc
aTghue9q4HsCzPhES98xloFKgVrmfx/w+Yxorcm9lYWK37PGHIME4FT3u8z7
Vw5IcwolH3DO3Zv25IOdPEhORf5NHiJAqo/9cc5P8lA+sD9fcIQAkw5BzFo1
ClD6zIAiPkoA/9zLH/UeKAKhJBsl4Vyj0LLnuYASHM0lpjwZw/OD45M5b7YS
lNyJVlX4RoC0VW//XZXKUOzSEAs/cL8NlrgMLqjClA3TV6afBHg39fg20101
ELMwkn6DM4+H/OVOSXV4pE0eujBJgHO21vflvTWg8PAhUctpAnwZSoqWlUJh
cp+b36EZPO9KPrvvnEZBhKfpzSTOK9MWB/kLAfK3jD1dCAR4xtIknJ2GQd73
Wy3+ZAIcz6tR4VU+A+NDg7tU5whgtjQo/KTkDAj1Cdtt4Jz2pFShm18Lcpub
WWOoBFh4c7q4clsLsjOo5vfnCdARsukz+vssfEtUKzddIABfKUfD3yAd2H8r
boN/kQDbH423Obd1IMtX9FHuEgFYj+Y1GgvoQaaR+XzFMgEO8hybFXLXhy86
RWc8V/B5xbwKfcZ2HgRQeobcXwIc4h8oa6k4Dw+kEzSbVglgwPUtMWPlAqSz
t93u3iDAlteaj2e7EcggJ9hubeLn7xcQL1E3hjfXs+O0/hHAVkBn/E6WMayO
Bt/s2iKAYcuksqelCViXKoV3MhIB0TuR8vyvKSx/LdmIZCKC9CN5gWRHM0ja
tTcUmImwLS6479WwGXT6LwV27CBC/ggc0n5hDuJaz6+3sRKh8+3M9WOPLKA9
UHj+BhsRDomRHk+LXQKLynueGuxEWDxfYn6x9BIk8Hq4tXAQ4RE1XzS54TJQ
JyQdm3cRYe5oXNPSiiU0hD26VM9HBFpja7wDuy0YVvN+8dtLhBmj1TKqvS2Q
pqPMT/ETwUOYGTXssIUD56+Z1Arg53HxO70MtYOI/ULnq/cTQZIozerA6QCC
BnffeR8gQsYpt6PN/g5QHbWue/IgEXZEntc9NekAU8Qx7SohIvx27qvqaXcE
ncYM9KkIEaZ3RkglpjvDBHlHu7soEUJNnc6e5nGB4EP+GicOE0GU9PlWQJIL
VN4yVa0UIwLzG6Xuu6muwG22W6H8KBGursZd+tDiBl/mE46WnCDCzSsmdkaZ
3rCiu/nTT5oIDwQe1X86dh34C72ztGSIkPOIkrAYex3MjS04p2WJcHas6fB5
PR8Yrj6yeFiBCJMG0xs3CL6wxJZZuYizd8hD6UwzP9hjx+H46hQRFGtcDjC9
9gNT3sUvtkpEKPVCCMvl/jDg09FWoEqEyFcgWp4aCAvv5AO91YjA0VFbm8oX
BLvFHp8EdSLs4ZTglMsMAsOhu49+auDz0+geaS4NhvcKVxIOARHispwfJpBC
4d3v32Y5Z4mgxsy5zcMUBaTzLrvcdYiQdHl68JZVFLCVfHujrksEvuJYnfeN
UaBn/krtmx7eP7ea75pvNLypTxLZf4EIQuVi394z3QQCF9NXEs5ZXQ87xt1v
AotTQGqzARGIsggD5+hN0Oa3ZrlsRISVzHRmSt0t6Ao4TnlgSoQON83/fiTE
wdSH3BJnMyJIsYT7Mu+IB0YJHhtlcyIkuKkcy4uOB+zLysDni0RofOXy6Mqt
BHip3N249woRAhLO0bQq7kDbX5uY1GtEGFH5uky9cR+Es5ivmdsS4Q7/4TOO
XMkQrVquIWBHBM4abUPEOhl0Qhb/PLQnAo9emOE5hhQYWL/pUuREhG6uzOxA
y1RQyJHUdnAmwqvbiTmU+lTI0PggetSFCCeH4zy2eNLAMkzgW7krEdKEc27v
+5gG0//KL1S7EyFdazXkl3UG6OQbHPf1IMJLm7iDrD0ZUI4usSh6EuGhvawq
88kH4BGp8bLJiwiFZv1SeWyZsMLwUf6lDxGO/L2yJD+cBRw7fgt8DCLCWqx7
PteHXPAoyfx9P5gIscMX120t8uCTtuaASQgRZjWurfFO5UF6bOztz6F4/dWX
Mp8zFIAw677N8XAiFFvlYH+eFUJUWdtYfgQR/MPIYlmbhTCta9dgG0mEA2/m
21MuFEF5QqXXTBQR7Bk4BXMWikCeA/lFuYnvn9WRXXnniiG98lf7s1tE+JqT
9ftWcTGs6Mdle8cS4SNR0Iq+VQwtdz+Z/Y4jwvu2oSXrlhLQ3mX/Zv02Eei8
Q27G2qVwmSf+CXsyEXiX2Ww4Kyvg9OeTr7/jbGRpTfogWQlS2WPfnqUQIYo/
5PeHikrYPCLFYZpGhHOkspHK2idQqPLBNfsBES79m80Yn3wGtzcDojwyiWBe
dvrofd8q8H0lnIVmESG58edm387noH3+eu/UQyJkd792vXmwGkg2fBIn8vD8
yK3WGbOqgcEjbcg/nK8kzq47P6+BFpLjxU/5RDA0+mPluKMWEn0bb/kVEuHE
7fqAV7W1IBd3Zbq1mAjWLM/VHh+uh+BnBYUXKonw+IvYsolBE9j56jULP8H1
ST56/X5LE+irLH5awDlJycY+73gzCL06s53+jAieT7kHzu56AZ0js1d/VBNB
IJBscZfSAhUPkwKe1xDhk93UTW7XVki1UbkXXYv731U2JZvQCo6khLaj9bie
1kn7KOQ2YNs8cdCrCa+/XlZLi+MlLHaOKGDNRGgZFdaeynwJ32LD9fe8wD9v
b8Yww9cJT3d/CmlswfOme1Fcu6wTjI/4jm214/4xEOm2H38FWfrNGfe7iWB1
xukU4WY3fDt/c+xqD+4P2QWp0z+64ZCB4UHpN7jeRnu2VlV64JHRTOHbXvy+
HQfiUxZ74Ik5z1PGfiLEyzMtkQJ6ocPGtctvEPffzsOR3r19wGB7iuXMEL5P
x7MffVHtBy27LV2eYTyf7URcyU/64a1D+oenI0QYf5SEHct8D0OunWOzo/g8
yEl9uTEfYdZ33/yln/i+Ly4knrwyCJL+0/ISE3jeWVQ/TSocBLeAKv8/OG/0
ZNsukgaBHqS9nvyLCAUepB8fwoZgNew6S98MEWQ93QQ1aoaBK/7tQU0KESxv
mx1oM/oChglpNhxUIvhYFyyPlH+BlNs2RWM4/+yp+XGUeRQEEv9IBNCJMEj+
9yu8dRREUkQVqhaJoFJjnOem8RXPi2A90VUiKFWkRnrcGIemGf+qSZx1fAey
tafGQVPWZ2/RGhEaItH8a/o/QPfVf5OHN4ig94r5mLvwT7AmWgYf2cLzxVtk
UuHLBMSdQsoldpBAa4+dfsPrX8AZrsZNxFmA3ar8H+MUJL9R8i9jIcF7YrIC
0+kpyLkii0mykuBdQ+6AzqspqI4UGTvOQQLxpfDx+b5p+NbPyCbDQ4KbmodQ
ix0EuMa/5UnD+eJDxh+qRvj3FJv14We8JOiTTnz/J5sA84tLBbJ8JNDzmlVi
wN+hHftmVOQESKCZ7jSc7UcCWcc3LqeESPCEpYYzTYgCMZt3etUkSbC+7Ln2
NmMBktzuunIcJ0HXoaw45U8LkD12l/07zmrf3vWfY1+EmvrE82EnSHDb8tG0
c+QiTHokfeqQxe8ToKgX+S0B/EgZ1VIiAdP+2Et64X9A/3xq8F5lvN5dR17W
v/oDFi9S98/ifJTk9Gp85zJ4ZaRZxamSgG3bUcQpbRnyDTIm3mqQwK9pN29E
wwpstmcRDc6Q4CR1qIBVZA3YpB8mCGuRQHg0XWDEew32Zj88Po/z55ffLHxf
rcGJwGz35LMkoEVdaDnusg6WsrnzQ3ok8NS9VFDZtgEv8gv+XjIiQcPcktex
lC0IjCxltbcmgUVkkWSzPxPWrTTWL36VBDPHDz37VcaE8VHx9MP5Yja2O+Q7
E1Z92XOf5zUS2PtWuP/WYsYo8orHg+1JEP79m5H4gR2Y7cwr/SRXEqS6WR6Y
+s6CnT/3416LPwl21nO1LjSxY9nb3KbhASS4XuP8qWWNHSM1YAJYIAmull5v
YtDkwOLESvJ7gkjAR8r/j7GLA+vccHs+EEoC/oQXHyaGOTHlqtUBQhQJ6qu2
3ljs5sZE+fbu5btHgm/Og9KRfrzYWxZX5iGcj/48uOthDi92fbV1KTmJBAwL
J44svebFOn84DHInk4Cyo1vMnn8PZltel8SRRoIbF94kq7fuwYoQM3amhyQw
dM/zGtqzFxNzTd1aKCHBUq34TMKiANZnSaA9f0yCZqun8XcPC2K+Buo/vEpJ
IL9S8Pi9iSDWpTDVSi0jAeGo9S/2OkHM4Z98CKmSBKZHpDc9gvdhJamDvyeq
SXBpImF/I/cB7Gg7D/ljGwnecqaSWF0PYeYjrnz57Xg/SV0XzpQcwmIonYhn
Bwmagv/r7Jg8hE3u90nl7CTB2dsB58MshbHswEENvdck+BK/VbplLILtlktN
evUOr/fthNlEiSi29ohPsf4LCdSl7Q+a1olhHxIEIh8ukCCj2/bnPv5j2Akn
r19UUTIEB/992JB0EhMnb/zXcZgMtQY7P5nUnsSEPG8v3Rcjg6Pf1Zejn09i
XAElzKfEyXBfNlPEiFsOo976djTkGBkEUu1Ho83lsKclOm4ssmQIEHZ+Y/RT
DpOeEf59SIMMJ7WTF/Rn5TFx16c35nFeNlV+t3dZHhOiqu14pUkGm70Dyu92
KGBcf8z5HVEy7LDVv79wRAGjMt9VqTxNhlzz9CA5OwXs6ZG/N5T1yKCi639A
blQBk3H4tMPoIhloqoITgs9PYRFHIpuFLMhw0ebJb4fWU9jAtKzHHM4W9x6Z
pPaewnwdk4ZjL5PBttaVlD5xCmt2MnjUZkWG4XvHMqS4FTFt136QsidD6vUY
c3snReyB5I3ff3GOLLu73OCtiBFJUmU9DmRQH31wgBSiiN357w63nRMZglol
9xLvKWIDbud+ZLqSIZy15de5ekXMyrM3lMWbDNKKoVsB64rYU5kg2WGcd4or
uD5kUsK2aBJThdfJ8M9CWCGLQwkr9Io7p+lLhhuXkpWPHVDCCN5n9/kFkMGn
SMI6TkUJU5Vb7scCyeBxoVAoClXCbi+URHIHkWFgUmzR7KwSJu2zg1gRTAY2
qQ30tqkS5uv7umHyBt5fMTVIwl0J61bw+68qjAxRWqI/qNeVMP7fYofCwslA
rfqomRiohDX7xdwSjMTrve3ldTFaCeNQVFCdxdnM+hdzZJwSZvXnF6U2Ctdz
bfhA9F28fv/TZoYxZGDIv32VO0MJM1Za3Cl0kwyxE5lmDx8qYUXLhS1knKcu
dlzbzFPCtAMZxGJjyaBW53hZv1QJy1Cu/mwaR4aug/sFlCrxfleu3RaNJ8Nb
P+PEv8/w/oJeLrQmkOHOXxND5gYlTDiqsPfGbTL0rUmw8zcrYbUJ0fkad8jA
uvtyEHOrEvbtodaF1rtkMOT5phjYqYR5PRI/ciORDAqFtv+tdylhzE9Y1tXv
kcHa+flBqx5cr7bespYkvP6l2P6ad0pYZ3d5ROh9Mkg11dSV9Sth5h9uX1RP
JoOqHZ0e8kEJI312k17H+d9pLX/JT0pY+M/zzC0pZNhzxEWjbkAJ20OU/haS
Sobvtzs1DwwpYaXzu2rU0siwsuzkc3VYCVNfpcev4cx+QHg4bEQJ+8gwYPMi
nQzfuG2tgz8rYfbsNUohGWQYiTJjM/6ihK3wpnKpPSDDx9GBIeZRJezOAb/p
VZzVrtg1p+EscsS8pTmTDA+0oXEb5/oTSinBWWTA5i726Y4pYXqKAq6qD8kQ
33b6tzvO45p/0VWcRayM5Txwvn52jL85mwyfs6Ki9XDeYfiCGpRDhpwbMUQG
nLMssl+r5JJhRrnJIQM/X/rajey/OGuLyq+y4PzKxdqnKY8Mvvl1ZaZ4vRbX
Eb2gfDLc3VMXcAPvZy5YWESlgAxPW82do/B+I6K3l1dwnkn3irDD9dhzZ/J9
YyEZvGXiXwv/X6/UV8WBRfj+3vis0YzrqZ7zKFT5ERkulWN0GVzvj8U3TVZw
Vvt6bSoSn4fDU0fJxmIyuM+HiFbh81qtP7sdUEKGfWdja5rxeSa2S3xReozX
43CsoKAX1+cj6WZDKRkGpeRamF7j+oy+swwoI4MBTX8rAvfL+ESlvFI5GQoW
Hd6PtithLIseE/UVZDik660hgvste82g0b+SDARq1Oou3I+yTCfvKT4hg1Xm
x6CvNXj/fIvq9U/JgPa83Nz5BO//4BCv/zMyVPlOHPmvDO9fvI50qooM084K
xuXFSliZUsCDuud4/pU1MnXk4PVfWlusrSXDtRjiht4dvH7bb29968ggwVbh
nxGrhB3+r7VAoZ4MIelSWS+jlDD90HCD2gY8b2ui0Gf4/ubkMlbUNP/fr2k9
C7ZKWOM7Qkv0C7xe0y5alaUSNrDS/96khQwHvzuWnzVXwnaaPFhYbCVDqIeN
6z8dPC9YTqgqvCSDk1r1uWwpJeyeAo8+UyfuP11W5cNHlLDya8tWQziHpqss
Rh9Uwn68eBnp20WG6PjZ6EFOXE8v897aHjL8qDu59mxOEXPIUfsa8wavn3xZ
i21KEYt4K0wx7cXn/evCe+SrIlYvRub+/ZYMHfHidhq9ipjIaLjFqfd4vvv9
lIkrUsRWoGy2bgjPM59bzreNFDFez8S/N4fxvOrZb1utrYhJZ/uwm4/geV21
h6VRTRGz/6Mh8+czPu/vTVo2RxSxD+UD/opfyVAuNXjp+PIprIR3nblhggzb
d4mzv1JPYSbTF8QaqWT4sye5tLlZAau67HApmIbr81rCTOYJ/h59DElUp5NB
XOKoZ2KeAva2uXSlY54MRTe5JzliFDA0iaG/d4kMG9jgvd36Cpi0Wr3f11Uy
uC5Kinh/kcdY7x/s2dwxB6/Latr0puSwdvU5F61Dc9DIdv6PT4UM9qEu2WvC
ZA7Ugt+17w8UxzjufGCvMp2DGtFV14tm4pieLcfjMLM5EFHkffxYThzr5rr1
ff/FOfg0v2kSSzmCtTgF6ZlfngPRQM6P5+2PYOWCNmLvbObAx1xUUNtcDIu5
ceJLrfscLCnFmghdEMU6TVx9oj3mYChNv/K+tCj279hjLmPPOTj6RDDk8C5R
LHhEWIvmNQfc6qqDZ/JEMG+ZPdXHfOdgYn9Mm3ypMGY9sXo7N3gOdp++9fZ4
txCmovUGiYudg9lzst1rCvuxZjvFTZ24Ofz/uZdLs9v7MNWoRy2s8XNwbsEu
gufDPkytPULldsIcSJb8Z3zYdR+moaIin3h3Dj53vmJSLhLE4ES5eGrKHDTU
7+/XFRHAdPnucObnzQFvWpnuo/N82Fv51Xc2+XMgkXM6/J4oH6Zn7JwgUjAH
M6ePu71Y3oOdu3eGpahwDmQVzlgNF+7BzrNubBUXz8FI4ZUF7nVezGjDfami
Yg7y6jmbc5p5sEvTBl8bGubA+etKHk8/F2Yj0TkU1zgHpcnY7N8ILsz5P4X3
l5rm4O+y+xmfU1yY/7zAy7XmOWDWkujel8+JJW9MlCBtc0BJNeKSC+XA3vL5
Xu/pmoMPd281HzZgwz5ZzPz34DWuf1eN7xo7G/bloYWDS/ccRKLSNta9rNiM
iIYF+5s56C7P6nulxYoxSO/QNHiH+6HyxaPNMzsxNe0HrJ8/zYFgdSPWYrkD
w+LZGEsH5qA1uvETi+gOTK8vdD1wcA5uidt9H55lxiyMr9H2Dc9B8AVf9mVf
ZszXWnLY+ssc8NUyTamnMWGV/i35M+NzEEC5NH+OwoAdfPxT6Q95DuiO5wx+
kTfAUS7GtGluDliN3/S6ZG1AVetR71DKHKAtyZXPdDfgzJBHOQNtDmx9mGT8
ytfBbXv9wK6FOaiqzi7r8V2DliuCDBIrc9Bnfdfx7vG/sGO2RYiM89EaKn/0
9AoYXrdRe/oX94Ps+/3i+SswFffYV2FtDtzELJI8+VeAvf7ULLqJ13dV14HO
uQyXuY37LjFRoJ/jDnOl4G949PA34QAzBbiit39Rvy8BVTyT+SfO42a/N/oK
lyBK46emIwsFbLIzXcyll6DM1eP5dTYKuDnN/AjTX4SVroT029wUOKxSp3FZ
ex4ygl7ZtB6gwLXflp8/eOG5CfGloQcpEDjs/+ruQTKM7zSgq+H/Jz5LFGqS
x79Xiz0YC28+RIHK9HEO2WMkqK6fz20QpUD2Z87PtTQC9C0e+v5cggIpbMXV
usUz4PViWsz7GAVs0+h6xVdngDe6wk1WkgJ8O8zY3u2bgcs8SutPj1PAWPD6
J/eUaZiVvbC/Uhr/vBL99fjdKWDwCL1UokABVtYv3ITiSSg+hRU4nKLAnWsy
+bxuk6CzwUIUU6TAh4LTzylyk3DvTkpgkRL+ef9zMeeiJ+BARXlGvioF0jZe
yq5/+gFKhC/DWSgFIgg/irsrvsHYs9yDl4ECFWU5Su763yAswN5BEKOA425Z
W03qV+jaQV/KOI1/vlQ3w/TUVzA+wrInTZsC5N3b8psfR8HN9pTxPX0KcG/W
3uhS+wzpF/4NWZ+nwJPcVkHx2RHoUO29KH2BAnYFjsHPU0ZgD6+VVb8BBWoU
5n8r0YfhxasYZ3YTCsxXOJuW1A8B25GhsNhLFAiTOx8m7z8ACrtzt80vU0Di
ScatUrkBsN5wihG/gs+jQstMfPwTVA+vxb+2pMDdGLNfMaMf4dKtw2mMNhSw
+E9+xXrmPZTOXq8Id6TAFTuy9UOTdzAwqC5t4ESBMh3l908Ib2G9fcdzIWcK
aDap1ZmGvwXDB1n1rS4UsDa9E3XueS8s63S+XHOjgBHp/k6jo29A+NSdM+/c
KXC7brI3+E0P6ImY92R5UOACf3Hw7v96IPcvsU/FC/ejk+/A9dpu0C7f/SXA
hwL/8Q/sj7n4GrzSv14+60sBlpcvO1MYX0NWVPH3vX4UGOmU+EWu6gLqZZVf
df4UELjprWS/qwvS2K9RF4MokNdQqLs51glTblVMXhEUUL3+OSnBsB2MCirK
T0ZSYFbdSC5qpA3ahksMF3He+43NR9u6DTI0c7L9oynAfuXFgYM+raDHnaBw
4xZ+v+SD+zbPX0DDmZtjmrG4H4u91PdpvQCxoIiILZz/7RSrvT7WDBsT/n3R
8RRY/bShRWJthqoaO/uEOxRwt5T5bh3SCAcJ1mz6dylgqenvHrS/ERIOXK7i
TKTAqT0OXTtbG8DupuF60j0KGOx//NmbqQH2XtRIzUimwJGrZw9PF9VByOre
rkcPcL1DaokqIjVAkOZxdcykQDxZP8VppBpM7Ti5JbIo8JHFLbTjTjVI9zFa
VjykwCE5AVLX8nP4mUNffJ6L78cLUdnlgWdwfoCc6ZNHAc/z96miBs+geccs
ciofv1/y3zmH/qeQ4vn9dmMBngfVus27Pj4BbXh7uP0RBcRVEf0TlAqo8Xvd
G1FMgUjv3DNIUAUIl3d4YiUUYDg1PHqCpQL+8jS8eP2YAvr7QyJuSpRD5VSR
SV85BaSkGH7bRZQCT9yN6NEq/Px+Z/1z4cXAPBFqX/GcAnO9q7fqhIphRSVU
60Y1ngfY+5uj7Y/gBzmYRaSWAp96EA85lkdQaRB427mBAr/WolsdywohrzTA
TbWRAisvbZW+mxVC8rb/eY4mCsR88FH6zFgIQTV+u6qa8f1TVyP4d+eDNr9P
ynIrBQZzAuU0XHNBxeu6b28bBW6MDhXtHMsBqV5vs4ftuD9i6pyD9XOAN8SL
H3mJ79udocHDKtnwc9z94a0uPI+0dEuuyGZBSIlzEd87PA+ydnEb3E0Hj39O
MbM49/I8nUsQTQdbCyeHpj4KyF4RGH7blAY6bI5Hrd7jeppVmMlTU2Gvh13F
o08U4EiT86h3TwHWHts7/gMU6HZS3CnCkwLrh2zddQbxPPZiV9RvTIZfn2xk
5oYocHHz20YXazJUnbKukf9CgZ9vnCqHw+5BUaJVKvMoBXK1Ku6yTiZC+qyl
32ecDcTK4s/qJMKNzCtKIV8p8IU5urT84F3Q27B40TlOgRMlxLgMUgJMdZm8
MprG6ws4VcHGEAsnL/t9ODRDAZ2dBBni/VsQRkv/SsF5vtVCY+XwLeDfN7YY
T6BAn9KhF+EXboKep41YJ5kCrq+X3hxuiYZ0pijZpDn8fZpcmvO1jIZfmUXq
1hQK8FQq6pdvRkFo14zpKpUCXu3+N3J1ouCZoPtNuQUKEPwvMgRPhMP608T7
WzjrGLUlR2iGg86Zqpz3i3g+yQqt9eaGwaTHYp3rbwo0p7U8n3W+AXu6gmYK
V/C8cLcueCISAsEet87y/qMA8WPZCQnBAOhhLDWZwFlRlMOwpMgf9mT2Xn22
RYGFz7Hf78j6w5NXHIHnGKhAYxW6bm/kBz8FUkqjmanAunPqwvYTH5B6Wltr
tIMKyVb6Qjf1fCDo9EjHIRYqbNW7zQkQrwOPh+Doi51UEBk8TTgrdR20XuWx
LrFT4cefSFab+56QbPGSr5ODCszc/Loo0QPGKZMiSZxUiGnFDj8+4wGBAuKq
UruooDDLd6GT2R3K3Std7XnweubT40yeuQK3QNO7IQEqHHWp22RIdoT83+xr
qYJUIG1Ya+SLOoLMoLWk2T4qOJShFwPrHOBCIlP88H4qnHTVCXs6YQ93mAy1
R4SoIMdhEMBjYgcHJgv90g9RIfWtmIDDki1UtP9+ZC5MhQKZlY2CDFt4G/yQ
8bMIFS6wMRx0eX0Nds7PdHwWo4KVbxD7feGr8OC9Cj3jCBXoVv9e/Aq3hqOV
dw5ZiFMhsyOQMXzSCrSd5MK+HKXC6vtZQYPnlhDz/YbaqCQVqms8o4t8LsN2
7576MVkq/Ik0POMZZg5Jj52ms07i/d0bYjZlNwfhm817rshR4a6QkFxelhmg
qI3PV3kqSAe6dyp3mEJYfYXsN0UqCFn3lCiImwBX6ubVbCUq+EZMsLO8NoYc
b6N7lspUqHV6ZTjtYAwtUsuUbyr433/QEFK8ZASrhVjFd3UqfPvCIFNbfAES
ItLGcjSocJVdNnV6+zwIWhNYrTWpoHJfviH52nlQFkx0Hkeo8P2SybzAcX3w
vzd65AdGhfOqda+XpnRhh7uUWd5pKqjvslIXuaILaXrhMVfPUKH/5sHzp4d0
oJb5yK8fWlR4ZGjOcvzDWVgI8cr/qUMF85O9vnPTWrAzrIftuS4V/iOe+3HD
WQuEIoT8IvXw+aZ8urdNOQN6MX26ovr4+Xo/WHO2T0PRHYlFOwNcz1uT4fa6
GDQnhlspGFKhJe+MVfxLgI9JIz1MRlRYEor/oLiMwnpqTHaxMRUatKPPTHoh
YJrzU2vWjAqqtivFsVnq4JqnVNVgToUTQV3B5TQ1iChI3Bd3EfdL6cM/0zpq
UFmsTpO4RIVzkUH/zrKoAtPTBw9cLXE/20fzqT9Xgn1VNAY1KyrYOuyQfiSm
BLLV2u7s1lS4fTnpi89DRbCsX4LKq1Rof+Aitpp8CuraDMkUWypQEqaP33ks
D30dj83a7Kjw9UIr8CvIw2TnZnuiPRXahhKU/V7JAVdPZaqMIxUKPwsOvBs4
CY4fdmp6u1BBsyB/saNSGkI/XS0FVyooqmZxRdBOQPJgPQ/Pf7geVTdekxVP
QNtn+9lqN5zPKYhcGjwOe392JC15UuFpUWFMLnIMpCb517q8qDAkvIfw8ZEE
YFMeDmneVPiYkFjwg1MCPAgHVBV9cH/+6K5UIItDDz3wV4A/FRIHD0rpfxSD
8YUP53UCqEB8qu01ZyQGS0vijQKBuH8CW41fjBwG4b9Dd5qC8PmNnTycQRSF
wG1ZxbVQKowePqsfYisMn841vWq9QYWbZ/X5+AsPgWQ6GEWEUeFG6Ez0mRkh
+CZp7MYcQYWeG+6DpJCDgJr6FO6KpgK/Uz1r4eg+yMxdkxnAWarwbymj2j5Y
IES3psZQoS4l4ZBcviA8upH6RfAWFcKSjNey/QSA5XEdl1g8FWJ3yg1a6O0F
m3mN7Bmc5+bWOPt6+KBRrftYWQIVzm4dWXXU4QPXjyNnpO9QIevqneJe4z3w
/u9ysPI9KpTvy1s1TOCBo2cidq7hvLuKNYDzEA9EJO5Mb02igof1nUd9rrtB
7rDgcyyZCunacfzOn3dBur7qrH4aFWaYUy7JkDiAlt7puyudCmKBxx6oqHKA
zoQewwDO5SEqZhmJ7LDqd+XgxQdUMCuVSpw+zQbWeaEm1x5SIaPp94+94yzQ
QGSaOJyN35/+vpbhAgtwK9z1mMHZZWh4T+TLHdD5Jif+v1wq5EvuY/hRzQxH
Ftra/Qqo4G6mb/29hRHC1M+eVy6kgjE/zyv+s4wwcuvD2CrO739cZTwzxABx
+3/+Dn9Ehcm+JHPjuW107sz28YTHVHDTyOxMb9xEte7FN+mXUmEt0a/z2soG
mju6++yuMnyedg4WzBobqKGHqG1qORWmb37+aj20htZmnH6Q+4QKdmnRlzSU
/6KOQlRDq6dUMDxidrcudwUVLH7AeuAZFZpY/vhs71xBw6opwZlVVOCTnWdv
IPxBz/ZnWKXWUIEt73nVsYEldNUE22tSi+etOmnsnekSWjk29353HRV4v/p5
pI4totwEQO/VUyEtXODyOdoCOsYwJ5rQRIUjZ9P0Unzp6J349K86zVTIdhJx
kSqloQg3pLK8wN8PsffvF35S0UcH05lvtuD7IXhkZ+w1CuqujBLC2/H3iX8q
2+chCRVuJ+VrduB5xPj1B/9vIjqglXZpA2frzH6tNWMiqmhCehvcSYWQuDqU
bz8B3XJPfeL3Gs9b+tt3H0en0eolTUeFbiqkvC1wfmQ8jTqEEIUWcWbRVdNd
eD+Fvo3TTPJ6g+cVhXyg+NMvNKWIcP2/d1Q47XPXVmVlAtWWTDku2Ye/D8XR
gaQDE+hKlcYUAWf/3BaJc2d/opZtyWaO76lw5uvdxoTKcVR8VF352icqfN5m
+a1R9xX9cnWWfmgA3yc5+J7P+BVNmLlfNo6zJ/Vow2PzMZS2OLPPcgjfF34+
RJJtFG3kur9h/hl/n+wkVYhFI6hrmlo93xcq3BOK3dDfM4IePDDjMYTzAUP9
7MC4YTTqmNpPozEqBHQFlr0OGULPnZnu1P+O37fenVVxewCNoaXU7x+nwvMD
PKHMXANoSxZWTsK5p8jJ9c+RT6j0fN79uJ/4PtCyLKuNP6A8uZdtun7h+jhu
0+xX36HndFlNU6aocEtjtiFZ/h0as9Rw1naaCqYx5aPR19+if/T4ZLZmqID1
f9li336Dji1/2FAj4b+PXw7zsexGeYvC5tnIVOiNkoz3aX+N6l84MT2K86V2
m03Wo6/RtkcJfQEUKhgUX7b0Z+xClw1UOrSpeN6wM2dxBL5CZddma/houN+y
Mn820TvRAiOthzV0XN9NQ/bXoy/RsfXFxKh5Kkww9fWdm+xAeUoLo4wWqGB5
O+/egYV2NGbznwttEX+/OVqm34i3oc4VzcrHl6nwhHycfWS8GS0wd5FaxTly
sNLk+IFmdGybX7h3hQoXHZn5dK42ofoWfixOq7i/I3YekfnTgN5kFFs7tUaF
dQtG2mPdBrTt6QCVaZ0KFf65AlqF9agMs+xI4QbOSRHlZrZ1qHPVeK/3JhXi
5cXRiN5aNP/K3Vb0H/6+sP6I1DhVi/JUkx6Nb1HhzX+l54mCNegf62KffUw0
qNfqkOE9WoXKsJk6EXGuNu05l6j0DHWuY7jSyEyDcYi/ihk8RcfYbTBzFhr4
anL+CEupRHkbuBSP7KTBect9b/LaK1B929ZjSziPcyUnSi6Uo22N+3Yns+Hn
tYk7tLmUoct2vUzX2GlAOckv/d/TUlR2V+CKDAcNuC/pyfeuPEYLHYZ/9HPi
592e9qPmlKA3eZKesu6mwcvUFZaAI4/QLWav129xNs1i2D+dXYSG/jX4dpuH
Bj/lIjdUBYpQ35+72Lj20GCkKZBJdl8havf0ngMPPw1+M2rNdSfmouMFnqGD
OC/HBn++Wp+DXk4zSEkVoIHMJ+SMwlQ2ahy66+XefTRwVU4OHzd6iILevQP7
D9JgZ88vvUKvB2iLhqf8N5wvJJmeCnydgSqfNNDLEaJBCXph48yhDFRWYFfg
IWEaBNm+3VKYSEMPzSQOHj6M33f1/HZHVgqaNepBmsJZ7+v8lwe8Keje/gvb
xWI0aFbMV7FLSka5arlkJMRpwHZk06tJ4j66EZmYIHWMBl3nGeUo9++igX4e
BRScH1eUVwy33EGXnC80PpWkAZPzDW8G6m10zoBrRlaKBooel95dt01AnU9T
1+dxfirVGGBdHI9OKb7nrTlBg3+fe+6NU+LQrwcT4ZQMXs8BTf2/ibGo+W4P
iz84X+bY7ag1ewsdYLrg2SBLA3bHte64M7fQt2TOHBU5GpiMfWSx57yJNjXd
XdE4RYN8AZ0P5UNRqMIT913/cBa4mUeYNYxCq/LPi3co0iBetd/wcH4kWhrL
aYop06BAQrw0sCEcFQuhuDKq0CDwRFL7zvdhaL5Hf2QXzrYqmcmp5Btohtnd
Z9pqNLASsx7+qBqK8ui6d7Oo08A+wOThQbcQNFH9/Pc3OHN77q/LLQpGbx3m
ZD+nSYP3xcPPb4oFoX7zdxwNgAZ3NXpZjrj4o4Gs3o3CGA2MskfiZ5v80GAR
M7YFnEn3g346cPuhYUYHn6aeoYF3vX3st34fNMJl+5+DFg0ehF18HKLgg0ZH
ThspadPAZsn2iFj+dTTu+ZPfo2dp8ON8jv4oqzea0Hv/bIUODaJfftodtuGJ
3pnwywzVpcHo8fOzq3890Pu7NTUOnaMBrzvB05LDHc3yfh9jfwH/PaWi4eI9
FzQ7vnrklAENnDvTNs90OqO5BekSLIY0yP4p/H1rzQkt+mTdV2aEz28k6i1T
tCNaQsSEQoxpoHzuqPvEBwe0dFvcS9+EBu/Ofa6pE3VAK0/SeGmmNLAM6PWM
GLVDn+oOOnSY0SBciWAroGGHPr/W0HDfnAYYoeI7qdgWrb8ffkXBggaHTxR0
ibJfQ9vndxUFXaHB1pXDK0vrlmgn69KSniUNEi9JR37hskS7RL5oH7Ciwbka
HpUOiSvoG6N8Ups1Dc6cvbac5HMJfesSo550Fd9//bInf4ss0P5I58RrNjQ4
Vjf/0eHrRXTguawcky0+74NIrZW1OTrUuydmGGdLuR39gWVm6MjEynCJHQ0u
sT+YqVs1Rb/ufhmk60CD27znJN48NUGnvI1eJjrToPAdbwvrU0N0Jv4Ur40L
DT5e7zje9NAAJRQIOpx0pYFf5jhjd/IFlPJpYufQfzQgWEYcefFQH6UTX18u
dqNBjsOKB/uTc+jCdlmlvzsN3EeqluO69dA/J68bCnri+WhLk/nDrYuu6JoX
knAuX2CosQEddO2a6tILLxow2IitUALPolv3GR5YX6eBZFVM2Y11LZR1IflH
kR8NnsRtuysbnEYvGJeaOvnTQNXENeRxC4amVrf2SgbQ4A+Sd3NIGkOFfAg1
zwPx/ROpac1FUNRucEPCL4gG1z9YbpdpImipPG+ucjANDqCqthzamqjckkZc
ewgN5q6HUnrd1NEAU5ONqFDcj6Zsq0W31dCWWufr2jdooB/s/2bguSp61i/F
sj+MBsaVXYPAq4Ja/iHIfo3E5/vhm4ahmCL+Pm0W50bh/oh+f+ZJwCl0pp53
v200DT5lJntXf1RAvQI0mYkxOB+rOLuRLI/WfTYJrryJ96/FE09ak0NXlVxo
nrfw/JnWYfvgIofeXEkZ/RNLA67lXffqimXRtxZlF5rjaPDFe93MNUcG5W5q
e3UjHve/0xzXSI40mhVEfMJ0mwYSl4h76xqk0KermlE8ibifrgW7IzrH0MXL
psvDOP/Sav2rEyaBKr9wccu8R4O/8pPLzC+Oop0hqebC92lg55V4q/WsOMry
rezdL5xfd4pzXcg4guqrt6OPk/H7rziPtc6JoZ/XiZLSqfi+yLNJBZUeRg9Y
/cubx1k7MT7Uieswatu6h68ujQbE3MZ7wUGi6NwN5J9aBg3E9L0nupuF0a3N
1EHdLPw95C440//iAKp1tVyH8yENkMTYLtFP+9GE9vbWjzhf+BmqeJm6D+WL
ID2+mEMD1mC1GSdEEL3889+B/bn4PmlVj7dfF0DzUb7kcZx/HNt1cKOSH5Xc
RkId8mlAkw1LQ0/tRT2vmc1LFNBAp/0L45coPrT2pavjHM6DQxyMASN7UM2o
NEOfIhqMqUorbyfyoiaMZLHIEhrceWHNcTyMG13IE32NPqaBY8dAY7LvLjRJ
/bLDFs4/ryZWDvtwoe/9eovDy2hwEaN+6bnNgbrxbGsh5fh5r97uJxWxo+zP
lGc2cfY66pDW/4oN1SE8Fg+rxP187ujFEUFWdCbmR7fGEzyfUn4PhxvsRG+K
8Dtt4MxucPDLwm0WtPPyrcehz2jAfCmR3URgB6reby8RUo3rE2vELhPEiI65
ZL9RraEB4zUyTfcHAxq0Y8h5FedMu0+i6XoMaD1yuiyoDp/fiXfu4tZbiNm3
YF2Vevx9HiP/mTT6hywFVhNXcA5s5w+qubCJnKwWkQxsxOuR8pmwtl1HPl64
9FapCfeXcJbU48A1xIOc5LqMM+OTJZ7x9FWkUmyr3P8Fft68BIWdtIKIZ4wf
92ujQa7R6ZA7U7+RLvm9fQrtNFgzCd20kv+N2H0877aEc83A5De1uCUkn7W1
0uclDVJLlk8bnF5ENEuW9OU7aTD7eltbtGwB+Y5JURZwni6sDFPfs4DsC314
4noXrt+LpOjD8zSkUWCw/+RrGjx8ZiBJm6IiF+vYPOZxltmxfoQ+QUFSqUFP
vXpoINz/OubiHzKy65qFjOdbGnC8i9F5FkFAnm7c+yD9jgb+F4/lMdfPIvpZ
PZ5UnCm7YnyvLMwgCYOKVe79NLgynbOfL3wakfDyMDzxngacQ6yFzB+mkB6O
Evoczv+0u2hcR6YQJm2+k24f8fzOEWzJnJxECif1Px3/RAMDXa6Iet1JBMJj
vMk4q35xo2OuE0hY4+Jz10EaTMnLblqr/kBWJAfkXEZoMHnM1Xha7ytycI6T
S+UzPs/Xj376G40h2BNd4s4vuB/eOVADro0id2Re5pWO4vuhP2znkP0ZeU5f
Dw4co0GdXIdmafcIMvJc2VznKw2cbodqsfwdRkQUqjgI32jwQtkrb9Z3CNH+
TZ5t+E4Dj+HSnpHWQcSt/uir2HEaeFoopLRyDiL1ynlBR3/SgG5vbjkQ8QnR
00iccfxFgwpXmQS+iT7Ec7P3peIU7v9ER73JoXdIajtzzo5p/PtcXwzD0sBb
ZBxumJTM4PnT17ZkPfcGYWRskvafxf3duNd7Y8cbRKJriVWbgOvDRT/6n2QP
4qPt1jFNpEEyvTPQLuE18oCl9GEdiQaRriecz3d3IW1vfvnfJNNA1maO8TVr
F7Lz3JUTRyh4XjJVpqqVdCLZBnpZ9nQahDYSTD9fakNect/yU5jH/S9YnVeu
1orMfHppyLSA60P/PJEq0YLImqqwPFrE8/zFWTsG0WbEjM9v0meJBmE7mjOb
TjQhwSNVrad/00D83ux/NKwReW0h4fvrD/59kkPR2SmuHiEJ2hvULNOgkb3G
ZrWhDuH+micZvYLvZ1uD0RNaLXLZin9CdBWfl2nKMyX/GoR6bccF2w28n+Q2
zdYnzxDew9gxuU18Ps6RKY1hTxHlqRtMDP9oUJZLeZF25QkS5fi7uWCLBikp
f9yPKFUgj4+eTPfepoFQp8PHVYVypI/g5g0MdFAc/uSwrF6G8P83dXSCkQ5G
b1Icvrg8RtSlhBmfM9FB6O5/+v/uliC2lCvjEcx0WKkCgYHmYqTSczBVmIUO
rt6kfPETjxDUt3P7KhsdiD33dAeH85HSGzJbIux0yJojU7HKPIQ7NnvzF87H
DnKONN/NRX5m+q85cdIhn+VN3nGvbESn6NffY1x0GHKueXLN6yFSVWm4QsaZ
00yOd19wFhLRLvnbk5sOZTvfuT0vfYAQ3mQsntxNB6VdR7dm+zIQwwGmhUWc
45XU5w3+piPC0+PUAF46FKtPOka7pCEdbCmE8L10CDrkq9Ocl4xI7NmaAX46
3EVpFQZ/7yNJB92nGQXo8KaR86HxyyTERvbsZKwgHT7EmxIZfRORNyq1P3X3
0eHkjpfBo1Z3EdnTIj/Y99Nhz4nJFVHjO8iW2drXewfoYFjwZDXeIgFxvuo8
ZnQQr6/W32fOJR756Dz8hVeIDq0sHNH6MXFIfsiz4YxDdFirfHarZuAWghTY
fcgTpUNKpVlCbH80UlL+sf/aYTo0W0dsHhOKRrhqNfoOi9HhpsnUspdfFDLe
zd/7+Agdmuw553/+F4Fof4zpcRGng65n4H/2SDjydHT+9fGjdGB9GetlJBSG
hM2963wmQQfVbPuahvUQZOa38kvvY3T4aamRZbsWjFz4V9wuL0mHgvMWfGOM
wYgQT0RLw3E6XPllEdYsG4i0KSnUt0vToWo3V3s/3RcRh4LaSBn8/FXDfYYK
vkiiHlfNaVncDzp781nDfRBrK8KznpN04O40TeQ6dh3pdjR7Gi9Hh+Bkz/Vy
Tm9E2quz8pw8HW6wkP5kffZENqOyyz4o0EF8J5v6y0R3xPEOa+n9U3RwibcI
5AtxQ96n+ZeYKOLzELK7ynz9PyS31LDosxIdQk+p9AuHuSCa75lyfqjSYYff
SeZ5JkdEat1v5LkaHaYf5A6dOu6A7Jec3RWjTocDUw6OGZftkZWbvVESmnTY
fOPuqzBoi8zUqLas4lyh/t7kxX5bZHii4ncfQocWzXNjhGEbpFoj0dkbcH0n
C/+GY9ZI/n//Ck5jdJBZQAe8OK2Qe5leX/lO00HO70qg3fQVxO238YXmM/g8
ZURFpJovIeJP+BUYdeiwXXzxTQLRDNnzNc59GOeK1Jj6PC4zhJF1teSxLh3u
mX5zXlE3RX7afxPUP0eHq54f5yk1xsj75PMmQvr4/JfzDPN/GyGtHW136DhX
npo/ypJmiGQdKPiXeoEOH3njPIW3ziNxejwqTgZ0SDvAsfTgrT4SEBh9XcWQ
DtFq0mEahecQ0yGH6e9G+DwzZq2++usiu+5IvhM3o0PR8eeRFo+1kM2mh0x/
ceYY/fZwse8MMjfLofnOnA67R1e3ZtZOI29P06o8LeiAYVllZ30xpNHbhgSX
6OC8ftmOEgDI47xPh/dcxve17Z3/TkEUiVmrSW+8QocvSnv438dqINePHfmY
YInv993gMyqm6si1i+msVlZ0eCsodfjnCTVEsyYwdNuaDvX3vV6V7lBBpCaI
dYNX6SDFmCjWyqSM7N91hVZsQwdEhLQ6z6mErLhq2OrZ0uHRXNLid+QUMvPg
6cMDdnSQeCW3+dxJARnuPjRMxXnvHpOZHw/kkWpRBp0UBzrYx/Yk1/HJIfmG
PpEOjnSo3V3zV6VdFrkXNtWs5EQHvdgu3w57GcRtrPvEN2fcfyHtp7S/SSGX
dyo7PXWhA9oiJ8P45Diiq1iWH+GK+6OuKJeeIImIJ9/mPeJGh6kxKW1hNwlk
T8e6/jLOfWK1ciYeRxFGqvutXnc8b612h2eEiCM/dQ3/unvi+vjMOlyvE0Mk
an9ZqXvRoVA6uZUydRjxEgroZPPG/XR/d5zVwcPI1kLO7cfX6cBV9Ycsd14E
0bE6SffzoYPycXv+a7RDSFJPl+kZXzp4+l47m5cvhAhnkw9O+NEhLn2Rt+vE
AcR5R3jUM386jL+IeBawcz9S5cUzeyOADnXvEzqGaIIIqqVStS+IDv0iig4u
k/xI3LO+PUScbzhgSO7cXuSjoE1QQzC+n677Hvlv8yFXqbcw01A6vDKxk9HW
3YOUWOx7LHqDDv6jyk52IbwItfMJ+zzOlyOvznrW8yBhGUNDd8Px/ka7WIrj
uZHubScVywi8v95f56ZkdyFcbms5kpF04D9KjT08y4nkoqJOPVF0yMTs+PaG
sSMz5XXv0qLxfWdfe5BqxYZI8+nK2sfg87Sqni3VZUXaiJ5/t27i54V43ffQ
YkF2mDJaf7iF55Eer+wB0x3Ihbb0zpxYOsy8PFNy0Z0ZGU9uva0aj+fhvVKO
ilZGRHzDkL4zgQ5wdH8BYZEB8XCaMv2Mc69LH2VTjgHZVGMX8r2D//1FdTfb
5H+aidxys9yJdGikjMK7zXVNoWmLqif36LAuQnddKlrVfNoUHqR3H38/d18J
57ixoqmZWILNJuP+Sl+3KLv5R/O9bT97TOr/94+W2N60pGmltDQknE6H5Mxf
BVl8i5pz7PtzWzPocGrfZmrZsXnN0J/gdDkT1y/Q8XWpG1WTo85ZdjmLDj+G
l3PC58ia2fH3/qZk4/lXxTyxXETUlLKu75TNxfex7AOnfuKsZovc99v9eXR4
b529fvDxtKY+C5OZawEddklzox6kX5rfvkoKsRTRwRoG1FJNJzVZSjQ0WEvo
MKe0p7/L77umnJehJXsZHb4lXiL7841qWqnahXBW/q+h+g9mOozjAD5KpZJQ
2VWWk85VrtRCnj3zLKVSJEzqDyu/cqVz/RiHMk6XHyuXEGHmV7p0yJpMdMx1
K2W+3y8hFXK7HXX4jmFSsp4/X39+7n3v94dGEuvQidaIzzDdVPjEog73019h
tsWHgjJ1usKygUb20VZziee74FBB0YCVHN97IOG36ex7GO0jEPRiz2s2nfho
UEH9X4ex3EYapXaEjPDMVHBNWO28TRON+Es+GSLeO8je275pyxv8Tzp8vdWO
Sihu1wZubafR3xkr/ca6JjgatPYRS4X3Uvh9kVS/hDGrSOYP7Mrj8FbB1Xpo
aM4rL3uP82EIRAW2dXAdy67BvpNG1QHelY2SF9Dt537SoYtG+Vq23OD1DD64
E2zh1EMjr6qkFa6sMigNYlEL2GOMfmlzVSl8uWc8t7MX79mUr5LBkcDewQTm
1T4aUfXJzZKSImjrKtlZ9wXnl0Gs5Gx8DCunNB7sERqN5nLdTeIfQIXg+mU0
QaOJ1ICB8P5EqOHdbw8x06H4Xr27qpsP6c02nlZuOuRfE8LoV22AwSZ77vVH
6VD1IEPQRao4Fzecyn5epEODnlq7bsc4zg7nt2ErP+mQbDT0nx/XiXM3xn0h
dlmHLMyNw6HbesCfaD9OHLZMOdKWadMDbkZGpNzGPlK4a59iXQ8ID324OhM7
yzJNxlqigJf/uG0F9rQ5xWIOU4DBLjz8GZtUa1zkZRQQ/TEkuht1iOFC8FKc
KGAwrG/jYh8KO6Rh2lMgdtbB9Bi2sXDn6GsmBQSTZ8RnsUWihxOL5hTgjjwr
voK9puaDM/8rCZY6zrWWYPuWEIOBUSQQtl0zVmDzA/jHps+RYLIl7ehz7PG+
rR75J0kwJK/vasQOuvH0lW4vCVqrVw11Y+9rE+fN6QlwsHK7fR92UzhtUGkJ
8EJ6MPIb9i+wW106QIDiQsHkGPaYOFN5qYUA1vlClylsB67m6+laAohzxMJZ
7P3FyRe4UgKYZpc3L2JHJSVA1xwCJGU1LRmxvy3PpLDTCPAfUr54Tw==
      "]]}}},
  AlignmentPoint->Center,
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox["\"\[Epsilon]\"", TraditionalForm], 
    FormBox["\"f(\[Epsilon])\"", TraditionalForm]},
  AxesOrigin->{0.675, 0},
  AxesStyle->{},
  Background->None,
  BaseStyle->{},
  BaselinePosition->Automatic,
  ColorOutput->Automatic,
  ContentSelectable->Automatic,
  DisplayFunction:>$DisplayFunction,
  Epilog->{},
  FormatType:>TraditionalForm,
  Frame->False,
  FrameLabel->None,
  FrameStyle->{},
  FrameTicks->Automatic,
  FrameTicksStyle->{},
  GridLines->None,
  GridLinesStyle->{},
  ImageMargins->0.,
  ImagePadding->All,
  ImageSize->Automatic,
  LabelStyle->{},
  Method->Automatic,
  PlotLabel->None,
  PlotRange->{{0.675, 0.8850000000000002}, All},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]},
  PlotRegion->Automatic,
  PreserveImageOptions->Automatic,
  Prolog->{},
  RotateLabel->True,
  Ticks->{{{0.7000000000000001, 
      FormBox["0.7000000000000001`", TraditionalForm]}, {0.75, 
      FormBox["0.75`", TraditionalForm]}, {0.8, 
      FormBox["0.8`", TraditionalForm]}, {0.8500000000000001, 
      FormBox["0.8500000000000001`", TraditionalForm]}}, Automatic},
  TicksStyle->{}]], "Output",
 CellChangeTimes->{3.4028167665300817`*^9, 3.4028173306612625`*^9, 
  3.4028180932678366`*^9, 3.402822052190488*^9, 3.430545517476883*^9, 
  3.4305456233691487`*^9},
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell["The distribution exhibits a weak negative skew:", "Text",
 Editable->False,
 CellChangeTimes->{{3.412415579593686*^9, 3.4124156003835807`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", "\[Epsilon]net", "]"}]], "Input",
 CellChangeTimes->{{3.4028159321503024`*^9, 3.4028159377183084`*^9}}],

Cell[BoxData[
 RowBox[{"-", "0.2874567888595763`"}]], "Output",
 CellChangeTimes->{3.402816766960701*^9, 3.402817331051824*^9, 
  3.402818049254549*^9, 3.4028180935482397`*^9, 3.402822052470891*^9, 
  3.4305455302853007`*^9, 3.4305456239099264`*^9}]
}, Open  ]],

Cell["\<\
The coefficient of variation can be approximated by Equation 5.34:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.4124156138429346`*^9, 3.4124156331006255`*^9}, 
   3.4124158475489874`*^9}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"StandardDeviation", "[", "\[Epsilon]net", "]"}], "/", 
  RowBox[{"Mean", "[", "\[Epsilon]net", "]"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"1", "/", "\[Epsilon]bar"}], "-", "1"}], ")"}], 
  SqrtBox[
   RowBox[{"2", 
    RowBox[{"(", 
     RowBox[{"1", "-", "\[Rho]corr"}], ")"}]}]], " ", "cv\[Beta]"}]}], "Input",\

 CellChangeTimes->{{3.402820933331648*^9, 3.4028209572860928`*^9}}],

Cell[BoxData["0.022851451156121427`"], "Output",
 CellChangeTimes->{3.402820959969952*^9, 3.402822052621107*^9, 
  3.4305455418719616`*^9, 3.4305456242404013`*^9}],

Cell[BoxData["0.022360679774997897`"], "Output",
 CellChangeTimes->{3.402820959969952*^9, 3.402822052621107*^9, 
  3.4305455418719616`*^9, 3.4305456242604303`*^9}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Implications for network permeability", "Subsection",
 Editable->False,
 CellChangeTimes->{{3.4028191276051407`*^9, 3.4028191424264526`*^9}}],

Cell["\<\
By applying the Kozeny-Carman equation, we can estimate the influence of \
structural uniformity on the uniformity of flow through the network:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.412415668110968*^9, 3.41241572148772*^9}}],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"v", "=", " ", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "/", 
      RowBox[{"\[Beta]zT", "[", 
       RowBox[{"[", "2", "]"}], "]"}]}], ")"}], 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "\[Epsilon]net"}], ")"}], "2"], "/", 
     SuperscriptBox["\[Epsilon]net", "3"]}]}]}], ";"}], "\n", 
 RowBox[{"Histogram", "[", 
  RowBox[{"v", ",", " ", 
   RowBox[{"HistogramCategories", "\[Rule]", "30"}], ",", 
   RowBox[{"HistogramScale", "\[Rule]", "1"}]}], "]"}]}], "Input",
 CellChangeTimes->{{3.4028179874957438`*^9, 3.402817990830539*^9}, {
   3.402820031715187*^9, 3.4028200843408594`*^9}, 3.4028213895176115`*^9, {
   3.4028216566317024`*^9, 3.4028216586746397`*^9}}],

Cell[BoxData[
 GraphicsBox[{
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], RectangleBox[{0., 0}, {0.0001, 0.02}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.00010000000000000002`, 0}, {0.0002, 94.64}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.00019999999999999998`, 0}, \
{0.00030000000000000003`, 1783.1899999999998`}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.00030000000000000003`, 0}, \
{0.0004000000000000001, 3466.11}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.00039999999999999996`, 0}, \
{0.0005, 2525.3900000000003`}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0005, 0}, {0.0006000000000000001, 1225.4399999999996`}]}, \

   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0006000000000000001, 0}, \
{0.0006999999999999999, 523.7300000000004}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0007, 0}, {0.0008, 215.9799999999999}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0008, 0}, {0.0009000000000000001, 92.57999999999997}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0009000000000000002, 0}, {0.001, 38.10000000000002}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.001, 0}, {0.0011000000000000003`, 17.909999999999993`}]}, \

   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0010999999999999998`, 0}, \
{0.0012000000000000001`, 8.509999999999996}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0012000000000000001`, 0}, \
{0.0013000000000000004`, 3.9799999999999986`}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0013000000000000002`, 0}, {0.0014, 1.980000000000003}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0013999999999999998`, 0}, \
{0.0015, 1.1599999999999995`}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0015, 0}, {0.0016000000000000003`, 0.3799999999999999}]}, \

   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0015999999999999999`, 0}, \
{0.0017000000000000001`, 0.2599999999999999}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0017000000000000001`, 0}, \
{0.0018000000000000004`, 0.2799999999999999}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0018000000000000002`, 0}, {0.0019, 0.1700000000000003}]}, \

   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0018999999999999998`, 0}, {0.002, 0.04999999999999999}]}, \

   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.002, 0}, {0.0021000000000000003`, 0.05999999999999985}]}, \

   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0021000000000000003`, 0}, \
{0.0021999999999999997`, 0.03000000000000005}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0022000000000000006`, 0}, {0.0023, 0}]}, 
   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0023, 0}, {0.0024000000000000002`, 0.0399999999999999}]}, \

   {RGBColor[0.7771114671549554, 0.7981689173723965, 0.92304875257496], 
    EdgeForm[GrayLevel[0]], 
    RectangleBox[{0.0024000000000000002`, 0}, \
{0.0024999999999999996`, 0.010000000000000016`}]}},
  AlignmentPoint->Center,
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->None,
  AxesOrigin->{0., 0},
  AxesStyle->{},
  Background->None,
  BaseStyle->{},
  BaselinePosition->Automatic,
  ColorOutput->Automatic,
  ContentSelectable->Automatic,
  DisplayFunction:>$DisplayFunction,
  Epilog->{},
  FormatType:>TraditionalForm,
  Frame->False,
  FrameLabel->None,
  FrameStyle->{},
  FrameTicks->Automatic,
  FrameTicksStyle->{},
  GridLines->None,
  GridLinesStyle->{},
  ImageMargins->0.,
  ImagePadding->All,
  ImageSize->Automatic,
  LabelStyle->{},
  Method->Automatic,
  PlotLabel->None,
  PlotRange->{{0., 0.0025}, All},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]},
  PlotRegion->Automatic,
  PreserveImageOptions->Automatic,
  Prolog->{},
  RotateLabel->True,
  Ticks->{{{0., 
      FormBox["0", TraditionalForm]}, {0.0005, 
      FormBox["0.0005`", TraditionalForm]}, {0.001, 
      FormBox["0.001`", TraditionalForm]}, {0.0015, 
      FormBox["0.0015`", TraditionalForm]}, {0.002, 
      FormBox["0.002`", TraditionalForm]}, {0.0025, 
      FormBox["0.0025`", TraditionalForm]}}, Automatic},
  TicksStyle->{}]], "Output",
 CellChangeTimes->{3.402821458516827*^9, 3.4028220543836412`*^9, 
  3.4305460624505167`*^9}]
}, Open  ]],

Cell["\<\
The coefficient of variation of local flow rate is considerably greater than \
that of porosity:\
\>", "Text",
 Editable->False,
 CellChangeTimes->{{3.4124157482461967`*^9, 3.4124157869318237`*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"StandardDeviation", "[", "v", "]"}], "/", 
  RowBox[{"Mean", "[", "v", "]"}]}]], "Input",
 CellChangeTimes->{{3.4028201006042447`*^9, 3.40282014149304*^9}, {
  3.402820198505019*^9, 3.4028202008684177`*^9}, {3.402821646537187*^9, 
  3.4028216713228273`*^9}, {3.402821701566315*^9, 3.4028217024676113`*^9}}],

Cell[BoxData["0.331446785565064`"], "Output",
 CellChangeTimes->{3.4028220545538864`*^9, 3.4305461596602974`*^9}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
WindowSize->{616, 750},
WindowMargins->{{Automatic, 48}, {Automatic, -16}},
PrintingOptions->{"PrintingMargins"->{{54., 53.875}, {72., 72.}}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 20, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Creative"}, "NaturalColor.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[590, 23, 162, 3, 73, "Title"],
Cell[755, 28, 224, 3, 36, "Subtitle"],
Cell[982, 33, 148, 2, 29, "Subsubtitle"],
Cell[CellGroupData[{
Cell[1155, 39, 135, 2, 75, "Section"],
Cell[1293, 43, 320, 7, 47, "Text"],
Cell[CellGroupData[{
Cell[1638, 54, 890, 22, 64, "Input"],
Cell[2531, 78, 21596, 439, 257, 17385, 368, "CachedBoxData", "BoxData", \
"Output"]
}, Open  ]],
Cell[24142, 520, 160, 2, 29, "Text"],
Cell[CellGroupData[{
Cell[24327, 526, 781, 19, 62, "Input"],
Cell[25111, 547, 812, 20, 72, "Output"],
Cell[25926, 569, 617, 13, 68, "Output"]
}, Open  ]],
Cell[26558, 585, 169, 4, 29, "Text"],
Cell[CellGroupData[{
Cell[26752, 593, 305, 7, 41, "Input"],
Cell[27060, 602, 942, 25, 74, "Output"]
}, Open  ]],
Cell[28017, 630, 144, 2, 29, "Text"],
Cell[CellGroupData[{
Cell[28186, 636, 583, 14, 102, "Input"],
Cell[28772, 652, 624, 14, 70, "Output"]
}, Open  ]],
Cell[29411, 669, 179, 4, 29, "Text"],
Cell[CellGroupData[{
Cell[29615, 677, 220, 5, 41, "Input"],
Cell[29838, 684, 944, 25, 74, "Output"]
}, Open  ]],
Cell[30797, 712, 137, 2, 29, "Text"],
Cell[CellGroupData[{
Cell[30959, 718, 629, 14, 102, "Input"],
Cell[31591, 734, 605, 14, 70, "Output"]
}, Open  ]],
Cell[32211, 751, 126, 2, 29, "Text"],
Cell[CellGroupData[{
Cell[32362, 757, 684, 17, 112, "Input"],
Cell[33049, 776, 974, 25, 64, "Output"],
Cell[34026, 803, 633, 14, 60, "Output"]
}, Open  ]],
Cell[34674, 820, 303, 9, 47, "Text"],
Cell[CellGroupData[{
Cell[35002, 833, 573, 14, 54, "Input"],
Cell[35578, 849, 1035, 28, 74, "Output"]
}, Open  ]],
Cell[36628, 880, 119, 2, 29, "Text"],
Cell[CellGroupData[{
Cell[36772, 886, 507, 12, 52, "Input"],
Cell[37282, 900, 579, 13, 65, "Output"]
}, Open  ]],
Cell[37876, 916, 247, 5, 47, "Text"],
Cell[CellGroupData[{
Cell[38148, 925, 1426, 36, 169, "Input"],
Cell[39577, 963, 161574, 2633, 308, 108819, 1766, "CachedBoxData", "BoxData", \
"Output"]
}, Open  ]],
Cell[201166, 3599, 360, 10, 47, "Text"],
Cell[CellGroupData[{
Cell[201551, 3613, 1382, 37, 140, "Input"],
Cell[202936, 3652, 366, 5, 40, "Output"],
Cell[203305, 3659, 367, 5, 40, "Output"],
Cell[203675, 3666, 363, 5, 40, "Output"]
}, Open  ]],
Cell[204053, 3674, 308, 7, 48, "Text"],
Cell[CellGroupData[{
Cell[204386, 3685, 2754, 75, 277, "Input"],
Cell[207143, 3762, 4967, 88, 144, "Output"]
}, Open  ]],
Cell[212125, 3853, 1131, 29, 123, "Text"],
Cell[CellGroupData[{
Cell[213281, 3886, 3054, 77, 227, "Input"],
Cell[216338, 3965, 2893, 56, 249, 1090, 25, "CachedBoxData", "BoxData", \
"Output"]
}, Open  ]],
Cell[219246, 4024, 369, 9, 65, "Text"],
Cell[CellGroupData[{
Cell[219640, 4037, 3083, 81, 300, "Input"],
Cell[222726, 4120, 18379, 312, 263, 13181, 225, "CachedBoxData", "BoxData", \
"Output"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[241154, 4438, 188, 3, 75, "Section"],
Cell[241345, 4443, 734, 24, 65, "Text"],
Cell[CellGroupData[{
Cell[242104, 4471, 912, 25, 82, "Input"],
Cell[243019, 4498, 564, 15, 57, "Output"]
}, Open  ]],
Cell[243598, 4516, 388, 9, 47, "Text"],
Cell[CellGroupData[{
Cell[244011, 4529, 1274, 31, 82, "Input"],
Cell[245288, 4562, 486, 14, 57, "Output"]
}, Open  ]],
Cell[245789, 4579, 525, 18, 47, "Text"],
Cell[CellGroupData[{
Cell[246339, 4601, 529, 15, 52, "Input"],
Cell[246871, 4618, 584, 19, 47, "Output"]
}, Open  ]],
Cell[247470, 4640, 220, 5, 47, "Text"],
Cell[CellGroupData[{
Cell[247715, 4649, 1197, 28, 97, "Input"],
Cell[248915, 4679, 24270, 428, 261, "Output"]
}, Open  ]],
Cell[273200, 5110, 575, 16, 65, "Text"],
Cell[CellGroupData[{
Cell[273800, 5130, 810, 20, 82, "Input"],
Cell[274613, 5152, 175, 4, 40, "Output"],
Cell[274791, 5158, 112, 1, 40, "Output"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[274952, 5165, 179, 3, 75, "Section"],
Cell[275134, 5170, 294, 6, 47, "Text"],
Cell[CellGroupData[{
Cell[275453, 5180, 1918, 51, 127, "Input"],
Cell[277374, 5233, 6895, 121, 252, 3268, 61, "CachedBoxData", "BoxData", \
"Output"]
}, Open  ]],
Cell[284284, 5357, 302, 12, 29, "Text"],
Cell[CellGroupData[{
Cell[284611, 5373, 155, 4, 41, "Input"],
Cell[284769, 5379, 479, 11, 57, "Output"]
}, Open  ]],
Cell[285263, 5393, 554, 13, 85, "Text"],
Cell[285820, 5408, 1113, 33, 122, "Input"],
Cell[286936, 5443, 442, 12, 67, "Text"],
Cell[287381, 5457, 1467, 36, 102, "Input"],
Cell[288851, 5495, 272, 6, 47, "Text"],
Cell[CellGroupData[{
Cell[289148, 5505, 477, 15, 41, "Input"],
Cell[289628, 5522, 1348, 40, 78, "Output"]
}, Open  ]],
Cell[290991, 5565, 218, 5, 47, "Text"],
Cell[291212, 5572, 1618, 43, 137, "Input"],
Cell[292833, 5617, 608, 10, 101, "Text"],
Cell[CellGroupData[{
Cell[293466, 5631, 2034, 55, 216, "Input"],
Cell[295503, 5688, 13031, 224, 249, 7794, 136, "CachedBoxData", "BoxData", \
"Output"],
Cell[308537, 5914, 7695, 134, 249, "Output"]
}, Open  ]],
Cell[316247, 6051, 385, 10, 47, "Text"],
Cell[CellGroupData[{
Cell[316657, 6065, 2069, 51, 142, "Input"],
Cell[318729, 6118, 5588, 98, 256, "Output"]
}, Open  ]],
Cell[324332, 6219, 325, 6, 65, "Text"],
Cell[CellGroupData[{
Cell[324682, 6229, 3382, 96, 282, "Input"],
Cell[328067, 6327, 4618, 84, 264, "Output"]
}, Open  ]],
Cell[332700, 6414, 757, 20, 83, "Text"],
Cell[CellGroupData[{
Cell[333482, 6438, 3794, 106, 329, "Input"],
Cell[337279, 6546, 5477, 88, 243, "Output"]
}, Open  ]],
Cell[342771, 6637, 263, 5, 47, "Text"],
Cell[CellGroupData[{
Cell[343059, 6646, 2455, 52, 173, "Input"],
Cell[345517, 6700, 1691, 33, 238, "Output"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[347257, 6739, 172, 3, 75, "Section"],
Cell[347432, 6744, 175, 4, 29, "Text"],
Cell[CellGroupData[{
Cell[347632, 6752, 446, 12, 43, "Input"],
Cell[348081, 6766, 153, 3, 55, "Output"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[348283, 6775, 197, 3, 75, "Section"],
Cell[348483, 6780, 879, 26, 87, "Text"],
Cell[349365, 6808, 1311, 38, 162, "Input"],
Cell[350679, 6848, 304, 6, 47, "Text"],
Cell[350986, 6856, 1010, 26, 106, "Input"],
Cell[351999, 6884, 315, 8, 48, "Text"],
Cell[CellGroupData[{
Cell[352339, 6896, 896, 17, 65, "Input"],
Cell[353238, 6915, 41358, 685, 252, "Output"]
}, Open  ]],
Cell[394611, 7603, 214, 5, 47, "Text"],
Cell[CellGroupData[{
Cell[394850, 7612, 1768, 45, 182, "Input"],
Cell[396621, 7659, 39227, 693, 265, 32394, 579, "CachedBoxData", "BoxData", \
"Output"]
}, Open  ]],
Cell[435863, 8355, 149, 2, 29, "Text"],
Cell[CellGroupData[{
Cell[436037, 8361, 143, 2, 41, "Input"],
Cell[436183, 8365, 249, 4, 40, "Output"]
}, Open  ]],
Cell[436447, 8372, 206, 5, 29, "Text"],
Cell[CellGroupData[{
Cell[436678, 8381, 458, 13, 64, "Input"],
Cell[437139, 8396, 163, 2, 40, "Output"],
Cell[437305, 8400, 163, 2, 40, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[437505, 8407, 147, 2, 37, "Subsection"],
Cell[437655, 8411, 250, 5, 47, "Text"],
Cell[CellGroupData[{
Cell[437930, 8420, 744, 19, 64, "Input"],
Cell[438677, 8441, 5945, 139, 248, "Output"]
}, Open  ]],
Cell[444637, 8583, 208, 5, 29, "Text"],
Cell[CellGroupData[{
Cell[444870, 8592, 341, 6, 41, "Input"],
Cell[445214, 8600, 113, 1, 40, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
